import { createEntityAdapter, createSelector, createSlice } from '@reduxjs/toolkit';

import { Release } from '../../../../models/ui';
import type { RootState } from '../../../store';
import { getNestedSliceName } from '../../../utils';
import { selectYp, ypReduxNamespace } from '../model';
import { ypRequestAsyncThunk } from '../ypRequestAsyncThunk';
import { loadReleasesData } from './releaseData';
import { getNestedSliceSelector } from '../../../utils/nestedSlice';

// region Initial state

const name = 'releases';
const namespace = getNestedSliceName(ypReduxNamespace, name);

const releasesAdapter = createEntityAdapter<Release>({
   selectId: r => r.id,
});

const initialState = {
   data: releasesAdapter.getInitialState(),
};

export type ReleasesState = typeof initialState;
// endregion

export const fetchReleases = ypRequestAsyncThunk(`${namespace}/fetchMany`, 'getReleases', { reset: false });
export const fetchRelease = ypRequestAsyncThunk(`${namespace}/fetchOne`, 'getRelease');

export const releasesSlice = createSlice({
   name: namespace,
   initialState,
   reducers: {},
   extraReducers: builder => {
      builder.addCase(fetchRelease.fulfilled, (state, action) => {
         releasesAdapter.upsertOne(state.data, action.payload.response);
      });
      builder.addCase(fetchReleases.fulfilled, (state, action) => {
         const { response, meta } = action.payload;
         if (meta.reset) {
            releasesAdapter.removeAll(state.data);
         }
         releasesAdapter.upsertMany(state.data, response);
      });

      builder.addCase(loadReleasesData.fulfilled, (state, action) => {
         const { response } = action.payload;
         releasesAdapter.upsertMany(state.data, response.releases);
      });
   },
});

// region Selectors

export const selectReleasesStore = getNestedSliceSelector({
   name,
   initialState,
   parentSelector: selectYp,
});

const selectReleasesData = createSelector(selectReleasesStore, store => store.data);

export const selectReleases = createSelector(selectReleasesData, data =>
   releasesAdapter.getSelectors().selectAll(data),
);
export const selectRelease = createSelector(
   (_: RootState, releaseId: string) => releaseId,
   selectReleasesData,
   (releaseId, data) => releasesAdapter.getSelectors().selectById(data, releaseId),
);
// endregion
