import { createSlice, PayloadAction, createSelector } from '@reduxjs/toolkit';
import { BaseThunkAPI } from '@reduxjs/toolkit/dist/createAsyncThunk';
import { isEqual } from '@yandex-infracloud-ui/libs';

import type { RootState } from '../../../store';

import { selectYp, ypMerge, ypReduxNamespace, YpStagesStore } from '../model';
import { ypRequestAsyncThunk } from '../ypRequestAsyncThunk';
import { getNestedSliceName, getNestedSliceSelector } from '../../../utils/nestedSlice';

const name = 'stages';
const namespace = getNestedSliceName(ypReduxNamespace, name);

const initialState = {} as YpStagesStore;

type ChunkApi = BaseThunkAPI<RootState, any>;
function getProjectIds(thunkApi: ChunkApi, stageIds: string[]): (string | null)[] {
   const stages = selectStages(thunkApi.getState());

   return stageIds.map(stageId => stages[stageId]?.meta?.project_id ?? null);
}

export const getStages = ypRequestAsyncThunk(`${namespace}/get`, 'getStages', (thunkApi, params) => ({
   projectIds: getProjectIds(thunkApi as ChunkApi, params[0].objectIds),
}));

/**
 * @note всегда включать путь с `meta.id`
 * @example ('/meta', '/meta/id', e => [e.meta], e => [e.meta.id])
 * @note выключать fetchTimestamps для несхематизированных аттрибутов(`/annotations` и другие)
 */
export const fetchStages = ypRequestAsyncThunk(`${namespace}/fetch`, 'selectStages', { reset: false });
export const deleteStage = ypRequestAsyncThunk(`${namespace}/delete`, 'deleteStage');
export const doStageLocationAction = ypRequestAsyncThunk(`${namespace}/doStageLocationAction`, 'doStageLocationAction');
export const updateStageDisabledClusters = ypRequestAsyncThunk(
   `${namespace}/updateStageDisabledClusters`,
   'updateStageDisabledClusters',
);

export const overrideStageDisruptionBudget = ypRequestAsyncThunk(
   `${namespace}/overrideStageDisruptionBudget`,
   'overrideStageDisruptionBudget',
);

export const stagesSlice = createSlice({
   name: namespace,
   initialState,
   reducers: {
      deleteFromState(state, action: PayloadAction<string>) {
         delete state[action.payload];
      },
   },
   extraReducers: builder => {
      builder.addCase(fetchStages.fulfilled, (state, action) => {
         const {
            response: { values = [] },
            meta,
         } = action.payload;
         const { reset } = meta;
         if (reset) {
            state = {};
         }
         for (const item of values) {
            const stageId = item.meta!.id!;
            state[stageId] = ypMerge({ ...(state[stageId] ?? {}) }, item);
         }
         return state;
      });

      builder.addCase(getStages.fulfilled, (state, action) => {
         const { response: values, params } = action.payload;
         const { objectIds } = params[0];
         for (let i = 0; i < objectIds.length; i += 1) {
            const item = values[i];
            const stageId = objectIds[i];
            const newStage = ypMerge({ ...(state[stageId] ?? {}) }, item);
            const exist = state[stageId];
            if (!isEqual(exist, newStage)) {
               state[stageId] = newStage;
            }
         }
      });

      builder.addCase(deleteStage.fulfilled, (state, action) => {
         const { params } = action.payload;
         const stageId = params[0];
         delete state[stageId];
      });
   },
});

export const selectStages = getNestedSliceSelector({
   name,
   initialState,
   parentSelector: selectYp,
});

export const selectStage = createSelector(
   selectStages,
   (_: RootState, stageId: string) => stageId,
   (stages, id) => stages[id],
);
