import { createSlice, createSelector } from '@reduxjs/toolkit';
import { ypReduxNamespace, YpClusterStore, selectYp } from '../model';
import { ypRequestAsyncThunk } from '../ypRequestAsyncThunk';
import type { RootState } from '../../../store';
import { TPod, TPodSet } from '../../../../proto-typings';
import { YpLocation } from '../../../../models/api';
import { getNestedSliceName, getNestedSliceSelector } from '../../../utils/nestedSlice';
import { PodsFiltersParams } from '../../../../models/ui/yp/view';

const name = 'ypCluster';
const namespace = getNestedSliceName(ypReduxNamespace, name);

const initialState = {} as YpClusterStore;

const filterFunctions: Record<keyof PodsFiltersParams, (pod: TPod, filter: PodsFiltersParams) => boolean> = {
   podId: (pod, filter) => (filter.podId ? pod.meta?.id === filter.podId : true),
   podSetId: (pod, filter) => (filter.podSetId ? pod.meta?.pod_set_id === filter.podSetId : true),
   service: (pod, filter) =>
      filter.service
         ? pod?.labels?.deploy?.stage_id === filter.service || pod?.labels?.nanny_service_id === filter.service
         : true,
   podState: (pod, filter) =>
      filter.podState && filter.podState.length && pod?.status?.agent?.state
         ? filter.podState.includes(pod?.status?.agent?.state)
         : true,
   evictionState: (pod, filter) =>
      filter.evictionState && filter.evictionState.length && pod?.status?.eviction?.state
         ? filter.evictionState.includes(pod?.status?.eviction?.state)
         : true,
   schedulingState: (pod, filter) =>
      filter.schedulingState && filter.schedulingState.length && pod?.status?.scheduling?.state
         ? filter.schedulingState.includes(pod?.status?.scheduling?.state)
         : true,
   query: () => true,
};

const getFilterExpression = (filter: PodsFiltersParams) => (pod: TPod) => {
   const filterKeys = Object.keys(filterFunctions);
   for (let i = 0; i < filterKeys.length; i += 1) {
      const fnKey = filterKeys[i] as keyof PodsFiltersParams;
      if (!filterFunctions[fnKey](pod, filter)) {
         return false;
      }
   }

   return true;
};

export const getClusterPods = ypRequestAsyncThunk(`${namespace}/getClusterPods`, 'getPods', { reset: false });
export const getClusterPodSets = ypRequestAsyncThunk(`${namespace}/getClusterPodSets`, 'getPodSets', { reset: false });

export const ypClusterSlice = createSlice({
   name: namespace,
   initialState,
   reducers: {},
   extraReducers: builder => {
      builder.addCase(getClusterPods.fulfilled, (state, action) => {
         const {
            payload: {
               params,
               response: { values },
               meta: { reset },
            },
         } = action;

         const { location } = params[0];
         const pods = values as TPod[];

         if (!state[location]) {
            state[location] = {
               pods: [],
               podSets: [],
            };
         }

         if (reset) {
            state[location].pods = pods;
         } else {
            state[location].pods = state[location].pods.concat(pods);
         }
      });

      builder.addCase(getClusterPodSets.fulfilled, (state, action) => {
         const {
            payload: {
               params,
               response: { values },
               meta: { reset },
            },
         } = action;

         const podSets = values as TPodSet[];
         const location = params[0];

         if (!state[location]) {
            state[location] = {
               pods: [],
               podSets: [],
            };
         }

         if (reset) {
            state[location].podSets = podSets;
         } else {
            state[location].podSets = state[location].podSets.concat(podSets);
         }
      });
   },
});

const selectYpClusterStore = getNestedSliceSelector({
   name,
   initialState,
   parentSelector: selectYp,
});

const selectYpCluster = createSelector(
   selectYpClusterStore,
   (_: RootState, cluster: YpLocation) => cluster,
   (store, location) => store[location],
);

export const selectClusterPods = (state: RootState, cluster: YpLocation, filter?: PodsFiltersParams) => {
   const clusterPods = selectYpCluster(state, cluster)?.pods ?? [];

   if (filter) {
      return clusterPods.filter(getFilterExpression(filter));
   }

   return clusterPods;
};

export const selectClusterPodSets = createSelector(selectYpCluster, cluster => cluster?.podSets);
