import { createSlice, PayloadAction } from '@reduxjs/toolkit';
import { defaultYPLimits, selectYp, YPFiltersStore, YPLocationFilter, ypReduxNamespace } from '../model';
import { GetApiThunkOutput } from '../../../utils';
import type { RootState } from '../../../store';
import { YpLocation } from '../../../../models/api';
import { getClusterReplicaSet, getClusterReplicaSets } from './ypReplicaSet';
import { getMulticlusterReplicaSetQueryFromFilter, getReplicaSetQueryFromFilter } from '../../../../utils/yp';
import { getMulticlusterReplicaSets, getMulticlusterReplicaSet } from './ypMulticlusterReplicaSet';
import { getNestedSliceName, getNestedSliceSelector } from '../../../utils/nestedSlice';

const name = 'ypFilters';
const namespace = getNestedSliceName(ypReduxNamespace, name);

const initialState = {} as YPFiltersStore;

const prepareState = (state: YPFiltersStore, location: YpLocation) => {
   if (!state[location]) {
      state[location] = {
         replicaSets: {},
         multiclusterReplicaSets: {},
      };

      return;
   }

   if (!state[location].replicaSets) {
      state[location].replicaSets = {};
   }
};

type RequestParams = {
   location: YpLocation;
   limit?: number;
   filterExpression?: string;
   reset?: boolean;
};

function processResponse(
   state: YPFiltersStore,
   entityType: keyof YPLocationFilter,
   requestParams: RequestParams,
   ids: string[],
) {
   const { location, limit = defaultYPLimits[entityType], filterExpression = '', reset = false } = requestParams;

   const canLoadMore = Boolean(ids.length && ids.length === limit);

   prepareState(state, location);

   const filter = state[location][entityType][filterExpression];

   if (filter) {
      filter.canLoadMore = canLoadMore;
      filter.hasLoaded = true;

      if (reset) {
         filter.ids = ids;
      } else if (ids.length) {
         filter.ids = Array.from(new Set([...filter.ids, ...ids]));
      }
   } else {
      state[location][entityType][filterExpression] = {
         ids,
         canLoadMore,
         hasLoaded: true,
      };
   }
}

export const ypFiltersSlice = createSlice({
   name: namespace,
   initialState,
   reducers: {},
   extraReducers: builder => {
      builder.addCase(getClusterReplicaSets.fulfilled, (state, action) => {
         const {
            payload: {
               params,
               response,
               meta: { reset },
            },
         } = action;

         const [location, filterExpression, continuationToken, limit] = params; //eslint-disable-line
         const ids = response.values.map(replicaSet => replicaSet.meta?.id ?? '').filter(Boolean);

         processResponse(state, 'replicaSets', { location, filterExpression, limit, reset }, ids);
      });

      builder.addCase(
         getClusterReplicaSet.fulfilled,
         (state, action: PayloadAction<GetApiThunkOutput<typeof getClusterReplicaSet>>) => {
            const {
               payload: { params, response },
            } = action;

            const [location, id] = params;

            const filterExpression = getReplicaSetQueryFromFilter({ replicaSetId: id });
            const ids = [response?.meta?.id].filter((replicaSetId): replicaSetId is string => Boolean(id));

            processResponse(state, 'replicaSets', { location, filterExpression, limit: 1, reset: true }, ids);
         },
      );

      builder.addCase(
         getMulticlusterReplicaSets.fulfilled,
         (state, action: PayloadAction<GetApiThunkOutput<typeof getMulticlusterReplicaSets>>) => {
            const {
               payload: {
                  params,
                  response,
                  meta: { reset },
               },
            } = action;

            const [filterExpression, continuationToken, limit] = params; //eslint-disable-line
            const ids = response.values.map(replicaSet => replicaSet.meta?.id ?? '').filter(Boolean);

            processResponse(
               state,
               'multiclusterReplicaSets',
               { location: YpLocation.XDC, filterExpression, limit, reset },
               ids,
            );
         },
      );

      builder.addCase(
         getMulticlusterReplicaSet.fulfilled,
         (state, action: PayloadAction<GetApiThunkOutput<typeof getMulticlusterReplicaSet>>) => {
            const {
               payload: { params, response },
            } = action;

            const [id] = params;

            const filterExpression = getMulticlusterReplicaSetQueryFromFilter({ replicaSetId: id });
            const ids = [response?.meta?.id].filter((replicaSetId): replicaSetId is string => Boolean(id));

            processResponse(
               state,
               'multiclusterReplicaSets',
               { location: YpLocation.XDC, filterExpression, limit: 1, reset: true },
               ids,
            );
         },
      );
   },
});

const selectYpFiltersStore = getNestedSliceSelector({
   name,
   initialState,
   parentSelector: selectYp,
});

export const getYPFilter = (
   state: RootState,
   location: YpLocation,
   entityType: keyof YPLocationFilter,
   filterExpression = '',
) => {
   const filterPerLocation = selectYpFiltersStore(state)[location] ?? {};
   const filterByType = filterPerLocation[entityType];

   return filterByType && filterByType[filterExpression];
};
