import { createSelector, createSlice } from '@reduxjs/toolkit';
import { YpLocation } from '../../../../models/api';
import { YpAccountQuota } from '../../../../models/ui';
import { TAccount, TAccountSpec, TAccountStatus, TPerSegmentResourceTotals, TPodSet } from '../../../../proto-typings';

import type { RootState } from '../../../store';
import { getNestedSliceName, getNestedSliceSelector } from '../../../utils';
import { selectYp, YPQuotaStore, ypReduxNamespace } from '../model';
import { ypRequestAsyncThunk } from '../ypRequestAsyncThunk';

const name = 'ypQuota';
const namespace = getNestedSliceName(ypReduxNamespace, name);

const initialState = {
   quotas: {},
   userAccounts: {},
} as YPQuotaStore;

export const getAccountQuotaId = (accountId: string, location: YpLocation, segment: string) =>
   `${accountId}_${location}_${segment}`;

export const getUserAllowedAccounts = ypRequestAsyncThunk(
   `${namespace}/getUserAllowedAccounts`,
   'getUserAllowedAccounts',
);
export const getAccounts = ypRequestAsyncThunk(`${namespace}/getAccounts`, 'getAccountsByIds');
export const getAccountPodSets = ypRequestAsyncThunk(`${namespace}/getAccountPodSets`, 'getPodSets', {
   segment: '',
   accountId: '',
});

const isUseResources = (resources?: TPerSegmentResourceTotals) =>
   resources &&
   (resources.cpu?.capacity ||
      resources.disk_per_storage_class?.ssd?.capacity ||
      resources.disk_per_storage_class?.ssd?.bandwidth ||
      resources.disk_per_storage_class?.hdd?.bandwidth ||
      resources.disk_per_storage_class?.hdd?.capacity ||
      resources.internet_address?.capacity ||
      resources.memory?.capacity ||
      resources.network?.bandwidth ||
      (resources.gpu_per_model && Object.values(resources.gpu_per_model).some(gpuUsage => gpuUsage.capacity)));

const getAcctountResourceData = (spec: TAccountSpec, status: TAccountStatus, segment: string) => {
   const usagePerSegmentDict = status?.resource_usage?.per_segment ?? {};
   const usage = usagePerSegmentDict[segment];
   const hasUsage = isUseResources(usage);

   const limitPerSegmentDict = spec?.resource_limits?.per_segment ?? {};
   const limit = limitPerSegmentDict[segment];
   const hasLimit = isUseResources(limit);

   return { showAccount: hasUsage || hasLimit, usage, limit };
};

export const ypQuotaSlice = createSlice({
   name: namespace,
   initialState,
   reducers: {},
   extraReducers: builder => {
      builder.addCase(getUserAllowedAccounts.fulfilled, (state, action) => {
         const { response, params } = action.payload;
         state.userAccounts[params[1]] = response;
      });

      builder.addCase(getAccounts.fulfilled, (state, action) => {
         const {
            payload: {
               response,
               params: [location],
            },
         } = action;

         const accounts = response as TAccount[];

         accounts.forEach(account => {
            const { meta, status, spec } = account;
            if (status && spec) {
               const accountId = meta!.id;

               const segments = Object.keys(spec?.resource_limits?.per_segment ?? {});
               segments.forEach(segment => {
                  const key = getAccountQuotaId(accountId, location, segment);
                  const { showAccount, usage, limit } = getAcctountResourceData(spec, status, segment);

                  if (showAccount) {
                     const quota: YpAccountQuota = {
                        accountId,
                        canLoadMore: true,
                        isPodSetsLoading: false,
                        cluster: location,
                        name: meta!.name ?? meta!.id,
                        podSets: [],
                        segment,
                        resourceLimits: limit,
                        resourceUsage: usage,
                     };

                     state.quotas[key] = quota;
                  }
               });
            }
         });
      });

      builder.addCase(getAccountPodSets.fulfilled, (state, action) => {
         const {
            payload: {
               response: { lastPage, values },
               params: [location],
               meta: { accountId, segment },
            },
         } = action;

         const podSets = values as TPodSet[];
         const key = `${accountId}_${location}_${segment}`;

         state.quotas[key].podSets.push(...podSets);
         if (lastPage) {
            state.quotas[key].canLoadMore = false;
         }
      });
   },
});

const selectYpQuotaStore = getNestedSliceSelector({
   name,
   initialState,
   parentSelector: selectYp,
});

export const getYpQuotaUserAccounts = createSelector(selectYpQuotaStore, store => store.userAccounts);
export const getYpQuotas = createSelector(selectYpQuotaStore, store => store.quotas);

const getQuotaKey = (_: RootState, quotaKey: string) => quotaKey;
const selectYpQuotaByKey = createSelector(getYpQuotas, getQuotaKey, (quotas, quotaKey) => quotas[quotaKey]);

export const getYpAccountPodSets = createSelector(selectYpQuotaByKey, quota => quota?.podSets ?? []);
export const canLoadMoreAccountPods = createSelector(selectYpQuotaByKey, quota => quota?.canLoadMore);
