import { createSelector, createSlice, PayloadAction } from '@reduxjs/toolkit';
import { selectYp, ypReduxNamespace, YPReplicaSetsStore } from '../model';
import { ypRequestAsyncThunk } from '../ypRequestAsyncThunk';
import { GetApiThunkOutput } from '../../../utils';
import type { RootState } from '../../../store';
import { TReplicaSet } from '../../../../proto-typings';
import { YpLocation } from '../../../../models/api';
import { getNestedSliceName, getNestedSliceSelector } from '../../../utils/nestedSlice';

const name = 'ypReplicaSets';
const namespace = getNestedSliceName(ypReduxNamespace, name);

const initialState = {} as YPReplicaSetsStore;

export const getClusterReplicaSets = ypRequestAsyncThunk(`${namespace}/getClusterReplicaSets`, 'getReplicaSets', {
   reset: false,
});
export const getClusterReplicaSet = ypRequestAsyncThunk(`${namespace}/getClusterReplicaSet`, 'getClusterReplicaSet');

export const ypClusterReplicaSetSlice = createSlice({
   name: namespace,
   initialState,
   reducers: {},
   extraReducers: builder => {
      builder.addCase(
         getClusterReplicaSets.fulfilled,
         (state, action: PayloadAction<GetApiThunkOutput<typeof getClusterReplicaSets>>) => {
            const {
               payload: { params, response },
            } = action;

            const replicaSets = response.values as TReplicaSet[];
            const [location] = params;

            if (!state[location]) {
               state[location] = {};
            }

            replicaSets.forEach(replicaSet => {
               const id = replicaSet.meta?.id;

               if (id) {
                  state[location][id] = replicaSet;
               }
            });
         },
      );

      builder.addCase(
         getClusterReplicaSet.fulfilled,
         (state, action: PayloadAction<GetApiThunkOutput<typeof getClusterReplicaSet>>) => {
            const {
               payload: { params, response },
            } = action;

            const [location, id] = params;

            if (!state[location]) {
               state[location] = {};
            }

            if (response) {
               state[location][id] = response as TReplicaSet;
            }
         },
      );
   },
});

const selectYpReplicaSetStore = getNestedSliceSelector({
   name,
   initialState,
   parentSelector: selectYp,
});

const selectLocation = (state: RootState, location: YpLocation) => selectYpReplicaSetStore(state)[location];

export const selectYpReplicaSets = createSelector(
   selectLocation,
   (state: RootState, location: YpLocation, ids?: string[]) => ids,
   (replicaSetsByLocation, ids) =>
      ids && replicaSetsByLocation ? ids.map(id => replicaSetsByLocation[id]) : undefined,
);

export const selectYpReplicaSet = createSelector(
   selectLocation,
   (state: RootState, location: YpLocation, id: string) => id,
   (replicaSetsByLocation, id) => replicaSetsByLocation && replicaSetsByLocation[id],
);
