/*
экспортируем createRequestThunkGenerator — генератор кастомных thunk для запросов апи, подробности ниже
*/

import { AsyncThunk, AsyncThunkAction, AsyncThunkPayloadCreator, createAsyncThunk } from '@reduxjs/toolkit';
import { Observable } from 'rxjs';

import { ApiServiceName } from '../../models/api';
import { registerRequest } from '../../services/api/registerRequest';
import { NetworkAggregatorItem } from '../slices/network/aggregator';
import { networkSlice } from '../slices/network/network';

type getObservableParam<O extends Observable<unknown>> = O extends Observable<infer X> ? X : never;

type ApiMethod<Input extends Parameters<(...args: unknown[]) => unknown>, Output> = (
   ...input: Input
) => Observable<Output>;

type ApiFormat<MethodName, Input extends unknown[], Output> = Record<
   MethodName extends string ? MethodName : string,
   ApiMethod<Input, Output>
>;

interface GetThunkProps<Api, Meta> {
   api: () => Api;
   requestKey?: string;
   meta?: Meta;
}

interface ApiThunkOutput<Input, Output, Meta> {
   response: Output;
   params: Input;
   meta: Meta;
}

type ThunkAdapter<
   Api extends ApiFormat<keyof Api, any[], getObservableParam<ReturnType<Api[keyof Api]>>>,
   MethodName extends keyof Api,
   Input extends Parameters<Api[MethodName]>,
   Output,
   Meta
> = AsyncThunk<ApiThunkOutput<Input, Output, Meta>, Input, {}> & {
   (...params: Input): AsyncThunkAction<ApiThunkOutput<Input, Output, Meta>, void, {}>;
   withRequestKey(key: string): ThunkAdapter<Api, MethodName, Input, Output, Meta>;
   withApi(api: Api): ThunkAdapter<Api, MethodName, Input, Output, Meta>;
   withMeta(meta: Meta): ThunkAdapter<Api, MethodName, Input, Output, Meta>;
};

type getOutput<T extends ThunkAdapter<any, any, any, any, any>> = T extends ThunkAdapter<any, any, any, infer X, any>
   ? X
   : never;

type getInput<T extends ThunkAdapter<any, any, any, any, any>> = T extends ThunkAdapter<any, any, infer X, any, any>
   ? X
   : never;

type getMeta<T extends ThunkAdapter<any, any, any, any, any>> = T extends ThunkAdapter<any, any, any, any, infer X>
   ? X
   : never;

export type GetApiThunkOutput<T extends ThunkAdapter<any, any, any, any, any>> = ApiThunkOutput<
   getInput<T>,
   getOutput<T>,
   getMeta<T>
>;
export type GetApiThunkError<T extends ThunkAdapter<any, any, any, any, any>> = {
   error: any;
   params: getInput<T>;
   meta: getMeta<T>;
};

type MethodFunctionDef<Api, MethodName extends keyof Api> = (api: Api) => Api[MethodName];

type getMethodFuncName<Api, MethodFunc> = MethodFunc extends MethodFunctionDef<Api, infer X> ? X : never;

function getMethodName<Api extends object, MethodFunc extends MethodFunctionDef<Api, keyof Api>>(func: MethodFunc) {
   const apiProxy = new Proxy<Api>({} as Api, {
      get(_, name) {
         return name;
      },
   });
   return (func(apiProxy) as unknown) as getMethodFuncName<Api, MethodFunc>;
}

type getRawMetaData<
   MetaData extends Record<string, unknown> | ((thunkApi: any, params: any) => Record<string, unknown>)
> = MetaData extends Record<string, unknown>
   ? MetaData
   : MetaData extends (thunkApi: any, params: any) => Record<string, unknown>
   ? ReturnType<MetaData>
   : never;

let lastRequestId = 0;

/**
 * Возвращает обертку над thunk для конкретного апи
 * @param defaultApi геттер к апи, которое мы оборачиваем, тип берем у него
 * @example
 * const ypRequestAsyncThunk = createRequestThunkGenerator(() => ypApi);
 * const fetchPods = ypRequestAsyncThunk(`${namespace}/fetch`, 'getPods');
 * // в компоненте
 * fetchPods.withRequestKey('status.pods')({podSetId: 'id-1', location: 'sas'});
 */
export function createRequestThunkGenerator<
   // важно проверять только формат возвращаемого значения
   Api extends ApiFormat<keyof Api, any[], getObservableParam<ReturnType<Api[keyof Api]>>>
>(defaultApi: () => Api) {
   return function createThunk<
      MetaData extends
         | Record<string, unknown>
         | ((thunkApi: Parameters<AsyncThunkPayloadCreator<any>>[1], params: Input) => Record<string, unknown>),
      MethodName extends keyof Api,
      Input extends Parameters<Api[MethodName]>,
      Output extends getObservableParam<ReturnType<Api[MethodName]>>,
      MethodFunc extends MethodFunctionDef<Api, MethodName>
   >(name: string, method: MethodName | MethodFunc, defaultMetaData: MetaData = {} as any) {
      const getThunk = ({ requestKey, api, meta }: GetThunkProps<Api, getRawMetaData<MetaData>>) =>
         // примешиваем дополнительные параметры
         createAsyncThunk<ApiThunkOutput<Input, Output, getRawMetaData<MetaData>>, Input>(name, (params, thunkApi) => {
            const requestId = String((lastRequestId += 1));
            if (requestKey) {
               // регистрируем начало запроса
               thunkApi.dispatch(networkSlice.actions.addPending({ requestId, requestKey }));
            }
            const metaData =
               meta ??
               (typeof defaultMetaData === 'function' ? (defaultMetaData as any)(thunkApi, params) : defaultMetaData);
            return new Promise<ApiThunkOutput<Input, Output, getRawMetaData<MetaData>>>((resolve, reject) => {
               // оборачиваем запрос в registerRequest, внутри различные манипуляции
               const methodName =
                  typeof method === 'string'
                     ? method
                     : (getMethodName<Api, MethodFunc>(method as MethodFunc) as getMethodFuncName<Api, MethodFunc>);

               registerRequest(api()[methodName as MethodName](...params), requestKey).subscribe(response => {
                  if (requestKey) {
                     thunkApi.dispatch(
                        networkSlice.actions.addOk({
                           requestId,
                           requestKey,
                        }),
                     );
                  }
                  resolve({
                     response: response as Output,
                     params,
                     meta: metaData,
                  });
               }, reject);
            }).catch((error: { saveInStore: NetworkAggregatorItem<ApiServiceName> | undefined }) => {
               if (requestKey) {
                  thunkApi.dispatch(
                     networkSlice.actions.addError({
                        requestId,
                        requestKey,
                        error,
                     }),
                  );
               }
               return thunkApi.rejectWithValue({
                  error,
                  params,
                  meta: metaData,
               });
            });
         });

      /**
       * Адаптер над thunk, для подмешивания параметров через свойства.
       * Требуется для отделения параметров обертки от параметров запроса апи
       */
      function getThunkAdapter(
         thunk: AsyncThunk<ApiThunkOutput<Input, Output, getRawMetaData<MetaData>>, Input, {}>,
         params: GetThunkProps<Api, getRawMetaData<MetaData>>,
      ) {
         const thunkAdapter = ((...extParams: Input) => thunk(extParams)) as ThunkAdapter<
            Api,
            MethodName,
            Input,
            Output,
            getRawMetaData<MetaData>
         >;

         thunkAdapter.pending = thunk.pending;
         thunkAdapter.fulfilled = thunk.fulfilled;
         thunkAdapter.rejected = thunk.rejected;

         // ключ запроса, используется для группы логически связанных запросов
         thunkAdapter.withRequestKey = (requestKey: string) => {
            const newParams = { ...params, requestKey };
            return getThunkAdapter(getThunk(newParams), newParams);
         };

         // кастомное апи, используется для возможности брать измененное апи(например из контекста react)
         thunkAdapter.withApi = (api: Api) => {
            const newParams = { ...params, api: () => api };
            return getThunkAdapter(getThunk(newParams), newParams);
         };

         // метаданные, полезные в редьюсерах
         thunkAdapter.withMeta = (meta: getRawMetaData<MetaData>) => {
            const newParams = { ...params, meta };
            return getThunkAdapter(getThunk(newParams), newParams);
         };

         return thunkAdapter;
      }

      // по умолчанию без ключа и с дефолтным апи
      return getThunkAdapter(getThunk({ api: defaultApi }), { api: defaultApi });
   };
}
