import { ActionCreatorWithoutPayload, ActionReducerMapBuilder, AnyAction, Reducer, Slice } from '@reduxjs/toolkit';

function getSubLevel(action: AnyAction, namespace: string): string {
   const subActionName = action.type.slice(namespace.length + 1);
   return subActionName.slice(0, subActionName.indexOf('/'));
}

export function checkSubLevel<T extends AnyAction>(namespace: string, initialState: Record<string, any>) {
   return (action: AnyAction): action is T => {
      if (!action.type.startsWith(namespace)) {
         return false;
      }
      const subLevel = getSubLevel(action, namespace);
      if (subLevel && initialState.hasOwnProperty(subLevel)) {
         return true;
      }
      return false;
   };
}

export function getSubLevelReducer<State>(namespace: string, reducers: Partial<Record<keyof State, Reducer>>) {
   return (state: any, action: AnyAction) => {
      for (const level of Object.keys(reducers) as (keyof State)[]) {
         const oldState = state[level];
         const newState = reducers[level]!(oldState, action);
         if (newState !== oldState) {
            state[level] = newState;
         }
      }
   };
}

type InitialState<T extends Slice> = T extends Slice<infer X, any, any> ? X : never;

export function getSliceInitialState<T extends Slice>(slice: T): InitialState<T> {
   return slice.reducer(undefined, { type: '' });
}

type NestedInitialState<S extends Record<string, Slice>> = {
   [N in keyof S]: InitialState<S[N]>;
};

export type GetSliceName<N extends string> = N extends `${string}/${infer X}` ? GetSliceName<X> : N;

type NestedReducers<S extends Record<string, Slice>> = {
   [N in keyof S]: S[N]['reducer'];
};

export function getNestedSliceData<
   N extends string,
   S extends { [K in keyof S & string]: Slice<any, any, `${string}/${K}`> }
>({ prefix, slices }: { prefix: N; slices: S }) {
   const initialState = {} as NestedInitialState<S>;
   const reducers = {} as NestedReducers<S>;
   for (const [name, slice] of Object.entries(slices) as [keyof S, S[keyof S]][]) {
      initialState[name] = getSliceInitialState(slice);
      reducers[name] = slice.reducer;
   }
   return {
      initialState,
      reducers,
      setupMatcher: (builder: ActionReducerMapBuilder<NestedInitialState<S>>) => {
         builder.addMatcher(
            checkSubLevel<ActionCreatorWithoutPayload<N>>(prefix, initialState),
            getSubLevelReducer<NestedInitialState<S>>(prefix, reducers),
         );
      },
   };
}

export function getNestedSliceName<P extends string, N extends string>(prefix: P, name: N): `${P}/${N}` {
   return `${prefix}/${name}` as `${P}/${N}`;
}

export type NestedSliceSettingsParams<N, S, ROOT, PS> = {
   name: N;
   initialState: S;
   parentSelector: (state: ROOT) => PS;
};
export interface SliceSettings<RootState, CurrentState> {
   sliceName: string;
   sliceSelector: (state: RootState) => CurrentState;
}

/** селектор для вложенного среза */
export function getNestedSliceSelector<N extends string, S, ROOT, PS extends { [name in N]: S }>({
   name,
   initialState,
   parentSelector,
}: NestedSliceSettingsParams<N, S, ROOT, PS>): (state: ROOT) => S {
   const sliceSelector: SliceSettings<ROOT, S>['sliceSelector'] = state => parentSelector(state)[name] ?? initialState;

   return sliceSelector;
}
