/*
Создаем объект с конфигурацией окружения: кластеры, сервисы, ...
Экспортируем хук для доступа к конфигу из контекста, сеттер и геттер для доступа вне реакта
*/

/* eslint-disable no-underscore-dangle */
import { INFRA_ENVIRONMENTS } from '@yandex-data-ui/infra-buzzer';
import { IListOption, YandexCookies } from '@yandex-infracloud-ui/libs';
import { memoize } from 'lodash';
import React, { useContext } from 'react';

import { ApiServiceEndpointName } from '../models/api';
// noinspection ES6PreferShortImport (circular)
import { BUZZER_SERVICES, DEFAULT_CLUSTERS } from '../models/constants';
import { EnvName } from '../models/environment';
// noinspection ES6PreferShortImport (circular)
import { Storage } from '../models/Storage';

const storage = new Storage();
export const configStorage = storage;

export class Config {
   private static env = Config.getEnvVariables();

   public static getReduxLoggerConfig() {
      const ignorePrefixes = ['@@redux-form', 'INFRA-BUZZER'];

      return {
         collapsed: storage.getItem('reduxLogger.collapsed', true),
         diff: storage.getItem('reduxLogger.diff', false),
         duration: storage.getItem('reduxLogger.duration', false),
         enabled: storage.getItem('reduxLogger.enabled', true),
         logErrors: storage.getItem('reduxLogger.logErrors', false),
         predicate: (getState: () => any, action: { type: string }) => {
            if (!action.type) {
               return true;
            }

            return !ignorePrefixes.some(p => action.type.startsWith(p));
         },
      };
   }

   public static getInfraBuzzerConfig() {
      const isDev = Config.env.NODE_ENV !== 'production';

      return {
         enabled: storage.getItem('infraBuzzer.enabled', true),
         environment: isDev ? INFRA_ENVIRONMENTS.TESTING : INFRA_ENVIRONMENTS.PRODUCTION,
         subscribeTo: BUZZER_SERVICES,
      };
   }

   public static getEnvVariables(): AppEnvVariables {
      return {
         NODE_ENV: process.env.NODE_ENV ?? 'development',
         PUBLIC_URL: process.env.PUBLIC_URL ?? '/',

         REACT_APP_METRIKA: window.REACT_APP_METRIKA ? parseInt(window.REACT_APP_METRIKA, 10) : undefined,
         REACT_APP_NOW: process.env.REACT_APP_NOW ?? '',
         REACT_APP_VERSION: process.env.REACT_APP_VERSION ?? 'unknown',
      };
   }

   public static isDeployAdmin(): boolean {
      return storage.getItem('is-deploy-admin', false);
   }

   public static ym({
      method,
      goal,
   }: {
      method: 'reachGoal';
      goal: 'old-logs' | 'new-logs' | 'old-logs-visits' | 'new-logs-visits' | 'show-new-logs' | 'hide-new-logs';
   }) {
      if (Config.env.REACT_APP_METRIKA) {
         if (method === 'reachGoal') {
            /* @ts-ignore */
            window.ym(Config.env.REACT_APP_METRIKA, method, goal);
         }
      }
   }

   public clusters: (IListOption & { default: boolean })[];

   // быстрая сортировка
   public clusterOrder: Record<string, number>;

   public readonly now: Date | null;

   public readonly version: string;

   public readonly user: AppUser;

   private _services: Map<ApiServiceEndpointName, string>;

   private _lostClusters: Set<string>;

   private yandexCookies = new YandexCookies();

   public get metrika(): number | undefined {
      return Config.env.REACT_APP_METRIKA;
   }

   public get rawConfig() {
      return this.config;
   }

   public get yasmPush(): string {
      return this.config.yasmPush;
   }

   public get clck(): string {
      return this.config.clck;
   }

   public get feedbackFormId(): number | null {
      return this.config.feedbackFormId ?? null;
   }

   public get serviceWorkerRefreshInterval(): number {
      return this.config.serviceWorkerRefreshIntervalMinutes * 60000;
   }

   public get idmSystemName(): string {
      return this.config.idmSystemName;
   }

   /**
    * @deprecated
    * Не используйте как признак, меняющий поведение или логику взаимодействия.
    * Функциональность UI не должа зависить от хоста.
    * Был инцидент SPI-29025
    */
   public get isLocalhost(): boolean {
      return window.location.host.includes('localhost');
   }

   public get envName(): EnvName {
      return this.config.envName as EnvName;
   }

   public get overrideGeoLocation(): AppConfig['overrideGeoLocation'] {
      return this.config.overrideGeoLocation;
   }

   constructor(private config: AppConfig, user?: AppUser) {
      this.now = Config.env.REACT_APP_NOW ? new Date(Config.env.REACT_APP_NOW) : null;

      this.version = Config.env.REACT_APP_VERSION;

      this.user = user ?? {
         uid: this.yandexCookies.getYandexUID(),
         login: this.yandexCookies.getYandexLogin(),
      };

      this.clusters = this.config.clusters.map(c => ({
         name: c.title,
         value: c.value,
         default: DEFAULT_CLUSTERS.has(c.value),
      }));

      this.clusterOrder = this.clusters.reduce((order, e, i) => {
         order[e.value] = i;
         return order;
      }, {} as Record<string, number>);

      this._services = new Map(this.config.services.map(r => [r.name as ApiServiceEndpointName, r.endpoint]));

      this._lostClusters = new Set(this.config.lostClusters);

      // TODO плохо, сервис пересоздаётся без необходимости
      setAppConfig(config);
   }

   public isClusterLost(cluster: string): boolean {
      return this._lostClusters.has(cluster);
   }

   public isShowNewLogs(): boolean {
      return storage.getItem('stages.showNewLogs', false);
   }

   public isStageAllowEditAny(): boolean {
      return storage.getItem('stages.allowEditAny', false);
   }

   public isUIDeveloper(): boolean {
      return storage.getItem('is-ui-developer', false);
   }

   public getDeployEngine(): string {
      return this.config.deployEngine || 'env_controller';
   }

   public showDevJson(): boolean {
      const key = 'dev.json.enabled';
      if (storage.hasItem(key)) {
         return storage.getItem(key, false);
      }

      return this.isLocalhost;
   }

   public showReloadStageWarning(): boolean {
      return storage.getItem('dev.reloadWarning.enable', true);
   }

   public getLogEnv(): Ya.Rum.LogEnvironment {
      switch (this.config.envName) {
         case 'ext-prod':
            return 'production';
         case 'ext-man-pre':
            return 'prestable';
         case 'ext-pre':
            return 'prestable';
         case 'ext-test':
            return 'testing';
         default:
            return 'development';
      }
   }

   public getApiEndpointForService(service: ApiServiceEndpointName, strict = true) {
      if (strict && !this._services.has(service)) {
         throw new Error(`Unsupported service "${service}". Add it to environment config!`);
      }
      return this._services.get(service) ?? null;
   }
}

let appConfig: AppConfig | null = null;

/**
 * @side_effects set appConfig
 */
export function setAppConfig(config: AppConfig) {
   appConfig = config;
}

/** Не вызывать на этапе сборки! */
export const getConfig = memoize(
   () => (appConfig ? new Config(appConfig) : null),
   () => appConfig,
);

export const ConfigContext = React.createContext<Config | null>(null);

export function useConfig(tolerant = false): Config | null {
   const config = useContext(ConfigContext);
   if (!tolerant && !config) {
      throw new Error('Expected value in ConfigContext');
   }

   return config;
}
