import { deepClone, isEmpty, isEqual } from '@yandex-infracloud-ui/libs';
import { firstValueFrom } from 'rxjs';

import { TDeployUnitSpec, TProject, TRelease, TStage, TStageSpec } from '../proto-typings';
import { ypApi } from './api';

// DEPLOY-4683
const recreationTimeout = () => new Promise(resolve => setTimeout(resolve, 60000));

// DEPLOY-5579
const ypRetryTimeout = () => new Promise(resolve => setTimeout(resolve, 10000));
const ypRetryAttempts = 20;

/**
 * Набор действий, которые код из E2Е тестов может дергать напрямую,
 * чтобы из тестов быстро использовать апи (без взаимодействия с интерфейсом)
 */
export class TestActions {
   public pushUrl?: (to: string) => void;

   public async projectExists(projectName: string): Promise<boolean> {
      return Boolean(await firstValueFrom(ypApi.getRawProject(projectName)));
   }

   public async createProject(project: TProject): Promise<boolean> {
      const projectName = project.meta!.id;

      const existProject = await firstValueFrom(ypApi.getRawProject(projectName));

      if (!existProject) {
         await firstValueFrom(ypApi.createProject(project));
         console.log(`Project "${projectName}" created`);
         return true;
      }

      if (!isProjectsEqual(existProject, project)) {
         await this.deleteProject(projectName);
         await recreationTimeout();
         await firstValueFrom(ypApi.createProject(project));
         console.log(`Project "${projectName}" recreated`);
         return true;
      }

      console.log(`Project "${projectName}" exists with the same spec, skip creation`);
      return false;
   }

   public async deleteProject(projectName: string): Promise<boolean> {
      try {
         let i = 0;

         /* eslint-disable no-await-in-loop */
         while (i < ypRetryAttempts) {
            const existProject = await firstValueFrom(ypApi.getRawProject(projectName));

            if (!existProject) {
               break;
            }

            await firstValueFrom(ypApi.deleteProject(projectName));
            await ypRetryTimeout();

            i += 1;
         }
         /* eslint-enable no-await-in-loop */
      } catch (err) {
         console.warn(err);
      }

      return true;
   }

   public async projectStagesExist(projectName: string): Promise<boolean> {
      const exist = await firstValueFrom(ypApi.selectStages({ project: projectName }));

      if (exist?.values && !isEmpty(exist.values)) {
         return true;
      }

      return false;
   }

   public async stageExists(stageName: string): Promise<boolean> {
      return Boolean(await firstValueFrom(ypApi.getRawStage(stageName)));
   }

   public async createStage(stage: TStage): Promise<boolean> {
      const stageName = stage.meta!.id;

      const existStage = await firstValueFrom(ypApi.getRawStage(stageName));

      if (!existStage) {
         await firstValueFrom(ypApi.createStage(stage));
         console.log(`Stage "${stageName}" created`);
         return true;
      }

      if (existStage.spec!.revision !== 1 || !isStagesEqual(existStage, stage)) {
         await this.deleteStage(stageName);
         await recreationTimeout();
         await firstValueFrom(ypApi.createStage(stage));
         console.log(`Stage "${stageName}" recreated`);
         return true;
      }

      console.log(`Stage "${stageName}" exists with the same spec, skip creation`);
      return false;
   }

   public async deleteStage(stageName: string): Promise<boolean> {
      try {
         let i = 0;

         /* eslint-disable no-await-in-loop */
         while (i < ypRetryAttempts) {
            const existStage = await firstValueFrom(ypApi.getRawStage(stageName));

            if (!existStage) {
               break;
            }

            await firstValueFrom(ypApi.deleteStage(stageName));
            await ypRetryTimeout();

            i += 1;
         }
         /* eslint-enable no-await-in-loop */
      } catch (err) {
         console.warn(err);
      }

      return true;
   }

   public async createRelease(release: TRelease): Promise<boolean> {
      const releaseName = release.meta!.id;

      try {
         const existReleases = await firstValueFrom(ypApi.getReleases([releaseName]));
         if (existReleases.length) {
            await this.deleteRelease(existReleases[0].id);
         }
      } catch (err) {
         console.warn(err);
      }

      await firstValueFrom(ypApi.createRelease(release));
      console.log(`Release "${releaseName}" created`);
      return true;
   }

   public async deleteRelease(releaseName: string): Promise<boolean> {
      try {
         let i = 0;

         /* eslint-disable no-await-in-loop */
         while (i < ypRetryAttempts) {
            const existReleases = await firstValueFrom(ypApi.getReleases([releaseName]));

            if (!existReleases) {
               break;
            }

            await firstValueFrom(ypApi.deleteRelease(releaseName));
            await ypRetryTimeout();

            i += 1;
         }
         /* eslint-enable no-await-in-loop */
      } catch (err) {
         console.warn(err);
      }

      return true;
   }
}

function isStagesEqual(fromApiStage: TStage, newStage: TStage): boolean {
   const existStage = deepClone(fromApiStage);

   // clean revision info from deploy units
   const duIds = Object.keys(existStage.spec?.deploy_units ?? {});
   duIds.forEach(duId => {
      const du = existStage.spec!.deploy_units[duId];

      delete (du as Partial<TDeployUnitSpec>).revision;
   });

   // clean spec
   if (fromApiStage.spec) {
      delete (existStage.spec as Partial<TStageSpec>)?.account_id;
   }

   const isSameMeta =
      existStage.meta!.project_id === newStage.meta!.project_id &&
      existStage.meta!.account_id === newStage.meta!.account_id;

   const isSameSpec = isEqual(existStage.spec, newStage.spec);

   return isSameMeta && isSameSpec;
}

function isProjectsEqual(fromApiProject: TProject, newProject: TProject): boolean {
   const existProject = deepClone(fromApiProject);

   return existProject.spec!.account_id === newProject.spec!.account_id;
}
