/*
экспортируем синглтон класса для контроля долгих запросов
getRequestWithTimeoutControl - обертка над запросами апи, которая следит за временем ожидания ответа сервера;
показывает тост для долгих запросов (LONG_REQUEST_PERIOD) и обрывает слишком долгие запросы (DEFAULT_REQUEST_TIMEOUT)
*/

import { Toaster } from '@yandex-cloud/uikit';
import { TIMES_IN_MS } from '@yandex-infracloud-ui/libs';
import React from 'react';
import { Observable } from 'rxjs';
import { DEFAULT_REQUEST_TIMEOUT, LONG_REQUEST_PERIOD } from '../../models';
import { RequestTimeoutData, RequestTimeoutItem, RequestTimeoutState } from '../../models/api';

// FIXME: тут сервис идёт в компоненты, подумать, куда вынести
import { TimeoutToast } from '../../components/network/TimeoutToast/TimeoutToast';

const toaster = new Toaster();

const TOAST_NAME = 'timeout';
const TOAST_PERIOD = 2 * TIMES_IN_MS.Minute;
const REQUEST_LIFE_TIME = 2 * TIMES_IN_MS.Minute;
const EMPTY_TOAST_TIME = 10 * TIMES_IN_MS.Second;

type RequestsStore = Map<string, RequestTimeoutItem>;

class TimeoutAggregator {
   private requests: RequestsStore = new Map();

   private requestIdsByKey = new Map<string, Set<string>>();

   private isExistToast = false;

   private toastTimeout = 0;

   private lastKeyIndex = 0;

   private lastTimeoutTimestamp = 0;

   constructor() {
      this.clear();
   }

   private clear() {
      const timestamp = Number(new Date());
      for (const requestTimestamp of this.requests.keys()) {
         if (timestamp - Number(requestTimestamp) > REQUEST_LIFE_TIME) {
            this.requests.delete(requestTimestamp);
         }
      }
      this.updateToast();
      const existLong = Array.from(this.requests.values()).some(e => e.state === RequestTimeoutState.Long);
      if (existLong) {
         this.emptyToastTimeout = window.setTimeout(this.updateToast.bind(this), EMPTY_TOAST_TIME);
      }
      window.setTimeout(() => this.clear(), REQUEST_LIFE_TIME);
   }

   private updateToast() {
      const content = this.createToastContent();
      if (content === null) {
         if (this.isExistToast) {
            toaster.removeToast(TOAST_NAME);
            this.isExistToast = false;
         }
         return;
      }
      if (this.isExistToast) {
         toaster.overrideToast(TOAST_NAME, {
            content,
         });
      } else {
         toaster.createToast({
            name: TOAST_NAME,
            allowAutoHiding: false,
            title: 'Time is out',
            content,
            isClosable: false,
         });
         this.isExistToast = true;
      }
      window.clearTimeout(this.toastTimeout);
      this.toastTimeout = window.setTimeout(() => {
         toaster.removeToast(TOAST_NAME);
         this.isExistToast = false;
      }, TOAST_PERIOD);
   }

   private createToastContent(defaultText?: string | null) {
      const longRequests = Array.from(this.requests.values()).filter(e => e.state === RequestTimeoutState.Long);
      if (longRequests.length === 0) {
         if (defaultText) {
            return defaultText;
         }
         if (Date.now() - this.lastTimeoutTimestamp < EMPTY_TOAST_TIME) {
            return 'No timeout request';
         }
         return null;
      }
      return React.createElement(TimeoutToast, {
         requests: longRequests,
      });
   }

   add({ url, requestKey, requestData }: RequestTimeoutData): string {
      const timestamp = Number(new Date());
      const key = String((this.lastKeyIndex += 1));
      const groupKey = requestKey ?? '';
      if (!this.requestIdsByKey.has(groupKey)) {
         this.requestIdsByKey.set(groupKey, new Set());
      }

      if (requestKey) {
         // прерываем предыдущие периодические запросы
         const oldRequestIds = this.requestIdsByKey.get(groupKey)!;
         for (const id of oldRequestIds.values()) {
            if (this.requests.has(id)) {
               this.requests.get(id)!.abort!();
               this.requests.delete(id);
            }
         }
         if (oldRequestIds.size > 0) {
            this.updateToast();
         }
         this.requestIdsByKey.get(groupKey)!.add(key);
      }

      this.requests.set(key, {
         timestamp,
         url,
         requestKey,
         requestData,
         aborted: false,
         state: RequestTimeoutState.Short,
      });

      window.setTimeout(() => {
         if (this.requests.has(key)) {
            const request = this.requests.get(key)!;
            request.state = RequestTimeoutState.Long;
            this.updateToast();
         }
      }, LONG_REQUEST_PERIOD);

      window.setTimeout(() => {
         if (this.requests.has(key)) {
            const request = this.requests.get(key)!;
            request.abort!();
            request.aborted = true;
            this.updateToast();
         }
      }, DEFAULT_REQUEST_TIMEOUT);

      return key;
   }

   addAbort(key: string, abort: () => void): void {
      if (this.requests.has(key)) {
         this.requests.get(key)!.abort = abort;
      }
   }

   private emptyToastTimeout = 0;

   delete(key: string) {
      let existLong = false;
      if (this.requests.has(key)) {
         existLong = Array.from(this.requests.values()).some(e => e.state === RequestTimeoutState.Long);
         const groupKey = this.requests.get(key)!.requestKey ?? '';
         if (this.requestIdsByKey.has(groupKey)) {
            this.requestIdsByKey.get(groupKey)!.delete(key);
         }
         this.requests.delete(key);
         if (existLong) {
            this.lastTimeoutTimestamp = Date.now();
            window.clearTimeout(this.emptyToastTimeout);
         }
      }
      this.updateToast();
      if (existLong) {
         this.emptyToastTimeout = window.setTimeout(this.updateToast.bind(this), EMPTY_TOAST_TIME);
      }
   }
}

export const timeoutAggregator = new TimeoutAggregator();

type RequestWithTimeoutControlProps<T> = RequestTimeoutData & {
   request: Observable<T>;
   timeoutAggregator: TimeoutAggregator;
};

export function getRequestWithTimeoutControl<T>({
   request,
   timeoutAggregator: aggregator,
   url,
   requestData,
   requestKey,
}: RequestWithTimeoutControlProps<T>) {
   return new Observable<T>(subscriber => {
      // добавляем информацию о запросе и получаем уникальный ключ запроса [1]
      const timeoutKey = aggregator.add({
         url,
         requestData,
         requestKey,
      });
      const subscription = request.subscribe(
         resp => {
            // запрос вернул результат, исключаем его из долгих запросов [2]
            aggregator.delete(timeoutKey);
            subscriber.next(resp);
            subscriber.complete();
         },
         error => {
            // запрос вернул ошибку, также исключаем его из долгих запросов [2]
            aggregator.delete(timeoutKey);
            subscriber.error(error);
         },
      );

      // добавляем функцию для прерывания запроса для слишком долгих запросов [3]
      aggregator.addAbort(timeoutKey, () => subscription.unsubscribe());

      /*
         так как в цепочке вложенных observable есть промис, пункты [2] помещаются в очередь микротасков
         так как [3] выполняется синхронно, то итоговая последовательность такова
         [1] -> [3] -> [2]
         таким образом, внутри timeoutAggregator запись запроса существует на момент [3]
      */
   });
}
