/*
Создаем и экспортируем обертки над экземплярами классов апи с нужной конфигурацией (описана в Config.ts)
Обертки являются основной точкой входа для использования апи в приложении
*/

import { memoize } from 'lodash';
import React, { useContext } from 'react';

import { ApiServiceEndpointName, YpLocation } from '../../models/api';
// noinspection ES6PreferShortImport (circular)
import { Config, getConfig } from '../Config';
import { AwacsApi } from './services/AwacsApi';
import { InfraApi } from './services/InfraApi';
import { InfraComponentsApi } from './services/InfracomponentsApi';
import { SandboxApi } from './services/SandboxApi';
import { SolomonApi } from './services/SolomonApi';
import { StartrekApi } from './services/StartrekApi';
import { YasmApi } from './services/YasmApi';
import { YpApi } from './services/YpApi';
import { XRayApi } from './services/XRayApi';
import { ProjectManagerApi } from './services/ProjectManagerApi';
import { InfraDoctorApi } from '../../modules/infra-doctor';

export { YpApi } from './services/YpApi';

export class ApiServices {
   public awacs: AwacsApi;

   public yasm: YasmApi;

   public yp: YpApi;

   public startrek: StartrekApi;

   public sandbox: SandboxApi;

   public infra: InfraApi;

   public infraComponents: InfraComponentsApi;

   public xray: XRayApi;

   public solomonApi: SolomonApi;

   public projectManager: ProjectManagerApi;

   public infraDoctor: InfraDoctorApi;

   constructor(config: Config) {
      this.awacs = new AwacsApi(config.getApiEndpointForService(ApiServiceEndpointName.Awacs)!);
      this.yasm = new YasmApi(config.getApiEndpointForService(ApiServiceEndpointName.Yasm)!);
      this.yp = new YpApi(
         {
            [YpLocation.XDC]: config.getApiEndpointForService(ApiServiceEndpointName.Yp, false),
            [YpLocation.SAS]: config.getApiEndpointForService(ApiServiceEndpointName.YpSas, false),
            [YpLocation.VLA]: config.getApiEndpointForService(ApiServiceEndpointName.YpVla, false),
            [YpLocation.MAN]: config.getApiEndpointForService(ApiServiceEndpointName.YpMan, false),
            [YpLocation.IVA]: config.getApiEndpointForService(ApiServiceEndpointName.YpIva, false),
            [YpLocation.MYT]: config.getApiEndpointForService(ApiServiceEndpointName.YpMyt, false),
            [YpLocation.SAS_TEST]: config.getApiEndpointForService(ApiServiceEndpointName.YpSasTest, false),
            [YpLocation.MAN_PRE]: config.getApiEndpointForService(ApiServiceEndpointName.YpManPre, false),
         },
         config.isLocalhost,
      );
      this.startrek = new StartrekApi(config.getApiEndpointForService(ApiServiceEndpointName.Startrek)!);
      this.sandbox = new SandboxApi(config.getApiEndpointForService(ApiServiceEndpointName.Sandbox)!);
      this.infra = new InfraApi(config.getApiEndpointForService(ApiServiceEndpointName.Infra)!);
      this.infraComponents = new InfraComponentsApi(
         config.getApiEndpointForService(ApiServiceEndpointName.InfraComponents)!,
      );
      this.xray = new XRayApi(config.getApiEndpointForService(ApiServiceEndpointName.XRayApi)!);
      this.solomonApi = new SolomonApi(config.getApiEndpointForService(ApiServiceEndpointName.Solomon)!);
      this.projectManager = new ProjectManagerApi(
         config.getApiEndpointForService(ApiServiceEndpointName.ProjectManager)!,
      );

      this.infraDoctor = new InfraDoctorApi(config.getApiEndpointForService(ApiServiceEndpointName.InfraDoctor)!);
   }
}

/**
 * получение апи сервисов для текущей конфигурации
 */
export const getApiServices = memoize(
   () => {
      const config = getConfig();
      if (!config) {
         throw new Error('Expected config');
      }

      return new ApiServices(config);
   },
   () => getConfig(),
);

export const ApiServicesContext = React.createContext<ApiServices | null>(null);

/**
 * получение конкретного апи по имени
 */
export function getApi<T extends keyof ApiServices>(name: T): ApiServices[T] {
   const api = getApiServices();

   return api[name];
}

// при желании можно избавится от прокси
/**
 * оборачивает получение апи в прокси
 */
function getApiProxy<T extends ApiServices[keyof ApiServices]>(apiGetter: () => T) {
   return new Proxy({} as T, {
      get<K extends keyof T>(_: T, name: string): T[K] | null {
         return name === '$$typeof' ? null : apiGetter()[name as K];
      },
   });
}

/**
 * Для нескольких react-контекстов с разными конфигурациями, без прямой необходимости не использовать
 */
export function useApiByName<T extends keyof ApiServices>(name: T): ApiServices[T] {
   const api = useContext(ApiServicesContext);
   if (api === null) {
      throw new Error('you must define value for ApiServicesContext');
   }

   return api[name];
}

/**
 * Возвращает апи awacs для текущего конфига
 */
export const getAwacsApi = () => getApi('awacs');
export const awacsApi = getApiProxy(getAwacsApi);

/**
 * Возвращает апи голована для текущего конфига
 */
export const getYasmApi = () => getApi('yasm');
export const yasmApi = getApiProxy(getYasmApi);

/**
 * Возвращает апи yp для текущего конфига
 */
export const getYpApi = () => getApi('yp');
export const ypApi = getApiProxy(getYpApi);

export function useYpApi() {
   return useApiByName('yp');
}

/**
 * Возвращает апи трекера для текущего конфига
 */
export const getStarTrekApi = () => getApi('startrek');
export const starTrekApi = getApiProxy(getStarTrekApi);

/**
 * Возвращает апи infra для текущего конфига
 */
export const getInfraApi = () => getApi('infra');
export const infraApi = getApiProxy(getInfraApi);

/**
 * Возвращает апи sandbox для текущего конфига
 */
export const getSandboxApi = () => getApi('sandbox');
export const sandboxApi = getApiProxy(getSandboxApi);

/**
 * Возвращает апи сервиса обновления инфраструктурных компонент sidecar-updater для текущего конфига
 */
export const getInfraComponentsApi = () => getApi('infraComponents');
export const infraComponentsApi = getApiProxy(getInfraComponentsApi);

/**
 * Возвращает апи сервиса x-ray для текущего конфига
 */
export const getXRayApi = () => getApi('xray');
export const xRayApi = getApiProxy(getXRayApi);

/**
 * Возвращает апи Solomon для текущего конфига
 */
export const getSolomonApi = () => getApi('solomonApi');
export const solomonApi = getApiProxy(getSolomonApi);

/**
 * Возвращает апи Project Manager для текущего конфига
 */
export const getProjectManagerApi = () => getApi('projectManager');
export const projectManagerApi = getApiProxy(getProjectManagerApi);

/**
 * Возвращает апи Infra doctor для текущего конфига
 */
export const getInfraDoctorApi = () => getApi('infraDoctor');
export const infraDoctorApi = getApiProxy(getInfraDoctorApi);
