/*
Обертка над запросом с функциями: хранение в redux данных запросов и ошибок, контроль долгих запросов и логи yasm
*/
import { Observable } from 'rxjs';
import { catchError } from 'rxjs/operators';
import { ApiErrorItemWithMetaData, ApiServiceName, RequestMetaData } from '../../models/api';
import { AggregatorAddProps, networkAggregator } from '../../redux/slices/network/aggregator';
import { ServiceErrorData } from '../../redux/slices/network/model';
import { ApiActionNames } from './actions';
import { requestWithLogError } from './LogError';
import { getRequestWithTimeoutControl, timeoutAggregator } from './TimeoutAggregator';

// изменить хак с прокидыванием коллбеками в либе
function getRequestMeta(request: Observable<unknown>): unknown {
   let current = request;
   let end = false;
   let meta = null;
   while (!end) {
      if (Symbol.for('requestMeta') in current) {
         end = true;
         meta = (current as any)[Symbol.for('requestMeta')];
      } else if ('source' in current) {
         current = current.source!;
      } else {
         end = true;
      }
   }
   return meta;
}

export function registerRequest<Service extends ApiServiceName, Action extends ApiActionNames[Service], Rsp>(
   request: Observable<Rsp>,
   requestKey?: string,
): Observable<Rsp> {
   type ErrorData = ServiceErrorData<Service>;
   type CurrentMetaData = RequestMetaData<Service, Action, Record<string, unknown>>;
   const meta = getRequestMeta(request);
   const { url, requestData } = (meta ?? {}) as CurrentMetaData;

   // функция для сохранения в redux store
   const requestWithErrorStore = request.pipe(
      catchError((error: ApiErrorItemWithMetaData<ErrorData, CurrentMetaData>) => {
         const { requestMetaData } = error;
         if (!requestMetaData || !requestKey) {
            Object.assign(error, { saveInStore: undefined });

            throw error;
         }
         const { service, action, url: requestUrl, requestData: data } = requestMetaData;
         const aggregatorData: AggregatorAddProps<Service, Action> = {
            requestKey,
            service,
            action,
            error: {
               body: error.body,
               resp: error.resp,
            },
            data: error.data,
            requestData: data,
            url: requestUrl,
         };
         const saveInStore = networkAggregator.add(aggregatorData);

         Object.assign(error, { saveInStore });

         throw error;
      }),
   );

   // контроль времени для долгих запросов
   const requestWithTimeoutControl = getRequestWithTimeoutControl({
      request: requestWithErrorStore,
      timeoutAggregator,
      url,
      requestData,
      requestKey,
   });

   // логи ошибок для yasm
   return requestWithLogError(requestWithTimeoutControl);
}
