/*
класс, описывающий высокоуровневые методы взаимодействия с awacs api
методы, которые используются непосредственно
*/
import { BaseApi, IApiError, unique } from '@yandex-infracloud-ui/libs';
import { Observable, of } from 'rxjs';
import { map } from 'rxjs/operators';
import { RawNamespaceItem } from '@yandex-infracloud-ui/awacs';

import { noop } from '../../../utils';
import {
   ApiServiceName,
   AwacsApiActions,
   AwacsApiParams,
   EndpointSetRecord,
   ListBalancersPayload,
   ListBackendsRsp,
   ApiRequest,
} from '../../../models/api';
import { getApiRequestParams } from '../requestParams';

export class AwacsApi extends BaseApi {
   public getNamespacesForEndpointSet(endpointSets: EndpointSetRecord[]): Observable<string[]> {
      if (endpointSets.length === 0) {
         return of([]);
      }

      const backends$: Observable<string[]> = this.listBackends(endpointSets).pipe(
         map(resp => unique(resp.backends.map(b => b.meta.namespaceId))),
      );

      return backends$.pipe(map((result: string[]) => unique(result)));
   }

   public ListBalancersByNamespace(namespaceId: string): Observable<ListBalancersPayload> {
      const data = { namespaceId };

      return this.apiRequest({
         action: AwacsApiActions.ListBalancers,
         data,
      });
   }

   public GetNamespaceMap(namespaceId: string): Observable<RawNamespaceItem[]> {
      return this.apiRequest({ action: AwacsApiActions.GetNamespaceAspectsSet, data: { id: namespaceId } }).pipe(
         map(response => {
            const json = response.aspectsSet.content.graph.content.inclusionGraphJson;
            return JSON.parse(json);
         }),
      );
   }

   protected getCSRF(): string {
      return '';
   }

   protected handleError(resp: Response, error: IApiError): void {
      noop();
   }

   private listBackends(endpointSets: EndpointSetRecord[]): Observable<ListBackendsRsp> {
      const data: AwacsApiParams[AwacsApiActions.ListBackends]['request'] = {
         query: {
            ypEndpointSetFullIdIn: endpointSets,
         },
      };

      return this.apiRequest({ action: AwacsApiActions.ListBackends, data });
   }

   private apiRequest<Action extends AwacsApiActions>({
      action,
      data,
   }: ApiRequest<ApiServiceName.AWACS, Action, 'request'>): Observable<AwacsApiParams[Action]['response']> {
      const { method, path, headers, credentials = 'include' } = getApiRequestParams({
         service: ApiServiceName.AWACS,
         action,
      });

      return this.request<void, AwacsApiParams[Action]['request'], AwacsApiParams[Action]['response']>({
         method,
         path: path.replace(/^\//, ''),
         body: data,
         headers,
         credentials,
         onlyJson: true,
      });
   }
}
