import { ApiCache, BaseApi, HttpMethod, IApiError } from '@yandex-infracloud-ui/libs';
import { Observable } from 'rxjs';
import { map } from 'rxjs/operators';
import { noop } from '../../../utils';

interface InfraRecord {
   id: string;
   name: string;
}

interface InfraServicePayload {
   id: number;
   name: string;
}

interface InfraEnvironmentPayload {
   id: number;
   name: string;
}

interface InfraEnvironmentInput {
   serviceId: number | undefined;
}

export class InfraApi extends BaseApi {
   private cache = new ApiCache();

   constructor(apiPrefix: string) {
      super(`${apiPrefix}/v1/`);
   }

   public getServices(): Observable<InfraRecord[]> {
      return this.cache.request('services', () =>
         this.request<void, void, InfraServicePayload[]>(HttpMethod.GET, 'services/mine').pipe(
            map(resp =>
               resp.map(v => ({
                  id: v.id?.toString(),
                  name: v.name,
               })),
            ),
         ),
      );
   }

   public getEnvironments(service: number | undefined): Observable<InfraRecord[]> {
      const params = { serviceId: service };

      return this.cache.request(`environments_${service ?? ''}`, () =>
         this.request<InfraEnvironmentInput, void, InfraEnvironmentPayload[]>(
            HttpMethod.GET,
            'environments/mine',
            params,
         ).pipe(
            map(resp =>
               resp.map(v => ({
                  id: v.id?.toString(),
                  name: v.name,
               })),
            ),
         ),
      );
   }

   protected getCSRF(): string {
      return '';
   }

   protected handleError(resp: Response, error: IApiError): void {
      noop();
   }
}
