import { Dictionary } from '@reduxjs/toolkit';
import { ApiCache, BaseApi, HttpMethod, IApiError } from '@yandex-infracloud-ui/libs';
import { Observable } from 'rxjs';
import { noop } from '../../../utils';

export type RevisionsMap = Dictionary<Array<string>>;

export enum SidecarType {
   PodAgentBin = 'POD_AGENT_BINARY',
   TVM = 'PORTO_LAYER_DEPLOY_TVMTOOL',
   Logbroker = 'LOGBROKER_TOOLS_PORTO_LAYER',
   DRU = 'DRU_LAYER',
   Patchers = 'Patchers',
   Xenial = 'PORTO_LAYER_SEARCH_UBUNTU_XENIAL_APP',
   Bionic = 'PORTO_LAYER_SEARCH_UBUNTU_BIONIC_APP',
   Focal = 'PORTO_LAYER_SEARCH_UBUNTU_FOCAL_APP',
}

export class InfraComponentsApi extends BaseApi {
   private cache = new ApiCache();

   constructor(apiPrefix: string) {
      super(`${apiPrefix}/api/`);
   }

   public getStatPatchers(): Observable<RevisionsMap> {
      return this.cache.request('stats-patchers', () =>
         this.request<void, void, RevisionsMap>(HttpMethod.GET, 'stat-patchers'),
      );
   }

   public getStatSidecars(sidecar: SidecarType): Observable<RevisionsMap> {
      return this.cache.request(`stats-sidecar-${sidecar}`, () =>
         this.request<void, void, RevisionsMap>(HttpMethod.GET, `stat-sidecars?sidecar=${sidecar}`),
      );
   }

   public applySidecar(sidecar: SidecarType, percent: number): Observable<string> {
      return this.request<void, { sidecar: string; percent: string }, string>(
         HttpMethod.POST,
         `apply-sidecar`,
         undefined,
         {
            sidecar,
            percent: `${percent}`,
         },
      );
   }

   public applyPatcher(revision: number, percent: number): Observable<string> {
      return this.request<void, { 'patchers-revision': number; percent: string }, string>(
         HttpMethod.POST,
         `apply-sidecar`,
         undefined,
         {
            'patchers-revision': revision,
            percent: `${percent}`,
         },
      );
   }

   protected parseError(body: any, resp: Response): { status: number; message: string } {
      return {
         status: resp.status,
         message: resp.status !== 200 ? resp.statusText : body,
      };
   }

   protected getCSRF(): string {
      return '';
   }

   protected handleError(resp: Response, error: IApiError): void {
      noop();
   }
}
