import { ApiCache, BaseApi, HttpMethod, IApiError } from '@yandex-infracloud-ui/libs';
import { Observable } from 'rxjs';
import { map } from 'rxjs/operators';
import { ResourceState, SandboxResource } from '../../../models/ui';
import { noop } from '../../../utils';

interface SandboxResources {
   items: SandboxResource[];
}

export interface SandboxResourceParams {
   type: string;
   state?: ResourceState;
   order?: string; // должен соответствовать цепочке полей SandboxResource: -time.created, +time.updated
   attributes?: Record<string, any>;
   limit?: number;
}

interface SandboxResourceReqParams extends SandboxResourceParams {
   attrs?: string;
}

export class SandboxApi extends BaseApi {
   private cache = new ApiCache();

   constructor(apiPrefix: string) {
      super(`${apiPrefix}/api/v1.0`);
   }

   public getResource(id: number): Observable<SandboxResource> {
      return this.cache.request(`resource_${id}`, () =>
         this.request<void, void, SandboxResource>(HttpMethod.GET, `resource/${id}`),
      );
   }

   public getResources({
      type,
      state = ResourceState.READY,
      order,
      attributes,
      limit = 10,
   }: SandboxResourceParams): Observable<SandboxResource[]> {
      const params: SandboxResourceReqParams = {
         limit,
         type,
         state,
         ...(order ? { order } : {}),
         ...(attributes ? { attrs: decodeURIComponent(JSON.stringify(attributes)) } : {}),
      };
      return this.cache.request(`sandbox_resource_${type}`, () =>
         this.request<SandboxResourceReqParams, void, SandboxResources>(HttpMethod.GET, `resource/`, params).pipe(
            map(e => e.items),
         ),
      );
   }

   protected getCSRF(): string {
      return '';
   }

   protected handleError(resp: Response, error: IApiError): void {
      noop();
   }
}
