import { BaseApi, HttpMethod, IApiError } from '@yandex-infracloud-ui/libs';
import { Observable, concat } from 'rxjs';
import { map, reduce } from 'rxjs/operators';
import { StartrekTicket } from '../../../models/ui';
import { noop } from '../../../utils';

type Issue = {
   key: string;
   summary?: string;
   status?: {
      display: string;
   };
   assignee?: {
      id: string;
   };
};

interface GetIssueParams {
   keys: string[];
   fields?: string[];
   perPage?: number;
   page?: number;
}

export class StartrekApi extends BaseApi {
   constructor(apiPrefix: string) {
      super(`${apiPrefix}/v2`);
   }

   public getTicketSummary(id: string): Observable<string> {
      return this.getIssueField(id, 'summary');
   }

   public getTicketsData(ids: string[]): Observable<StartrekTicket[]> {
      return this.getIssues(ids, ['summary', 'status', 'assignee']).pipe(
         map(issues =>
            issues.map(issue => {
               const { status, summary, assignee, key } = issue;
               const ticket: StartrekTicket = {
                  id: key,
                  ...(status ? { status: status.display } : {}),
                  ...(summary ? { summary } : {}),
                  ...(assignee ? { user: assignee.id } : {}),
               };
               return ticket;
            }),
         ),
      );
   }

   protected getCSRF(): string {
      return '';
   }

   protected handleError(resp: Response, error: IApiError): void {
      noop();
   }

   private getIssueField(id: string, field: string): Observable<string> {
      return this.request(HttpMethod.GET, `issues/${id}/${field}`);
   }

   private getIssues(ids: string[], fields?: string[]): Observable<Issue[]> {
      const limit = ids.length;
      const pageLimit = 100;
      const pageCount = (limit - (limit % pageLimit)) / pageLimit + (limit % pageLimit > 0 ? 1 : 0);

      const requests: Observable<Issue[]>[] = [];
      for (let i = 0; i < pageCount; i += 1) {
         const idsRange = ids.slice(pageLimit * i, pageLimit * (i + 1));
         requests.push(
            this.request<GetIssueParams, undefined, Issue[]>({
               method: HttpMethod.GET,
               credentials: 'include',
               onlyJson: true,
               path: 'issues',
               params: {
                  keys: idsRange,
                  perPage: pageLimit,
                  page: i + 1,
                  ...(fields ? { fields } : {}),
               },
            }),
         );
      }
      return concat(...requests).pipe(
         reduce((acc, e) => {
            acc.push(...e);
            return acc;
         }, [] as Issue[]),
      );
   }
}
