import { ApiCache, BaseApi, HttpMethod, IApiError } from '@yandex-infracloud-ui/libs';
import { Observable } from 'rxjs';
import { map } from 'rxjs/operators';
import { noop } from '../../../utils';

interface ITypeResponse {
   total: number;
   result: string[];
}

interface ITypePayload {
   fqdn: string;
   status: string;
   response: ITypeResponse;
}

interface RawTagKeysPayload {
   [key: string]: string[];
}

interface TagsResponse {
   total: number;
   result: {
      [key: string]: string;
   }[];
}

interface TagsPayload {
   fqdn: string;
   status: string;
   response: TagsResponse;
}

interface TagKeysInput {
   itype: string;
}

interface TagsInput {
   itype: string;
   fields: string;
}

export class YasmApi extends BaseApi {
   private cache = new ApiCache();

   constructor(yasmHost: string) {
      super(`${yasmHost}/metainfo/`);
   }

   public getITypes(): Observable<string[]> {
      return this.cache.request(`yasm_itype_list`, () =>
         this.request<void, void, ITypePayload>(HttpMethod.GET, 'tags/itype').pipe(map(resp => resp.response.result)),
      );
   }

   public getTagKeys(itype: string): Observable<string[]> {
      const params: TagKeysInput = {
         itype,
      };

      return this.cache.request(`yasm_itype_${itype}_tag_keys_list`, () =>
         this.request<TagKeysInput, void, RawTagKeysPayload>(HttpMethod.GET, 'tagkeys', params).pipe(
            map(resp => resp[itype]),
         ),
      );
   }

   public getTags(itype: string, field: string): Observable<string[]> {
      const params: TagsInput = {
         itype,
         fields: field,
      };

      return this.cache.request(`yasm_itype_${itype}_${field}_tags_list`, () =>
         this.request<TagsInput, void, TagsPayload>(HttpMethod.GET, 'tags', params).pipe(
            map(resp => resp.response.result.map(item => item[field]).filter(item => item !== '')),
         ),
      );
   }

   protected getCSRF(): string {
      return '';
   }

   protected getFetchParams(method: HttpMethod, body: string | undefined, headers: HeadersInit): RequestInit {
      return {
         body,
         headers,
         method,
      };
   }

   protected handleError(resp: Response, error: IApiError): void {
      noop();
   }
}
