import { IApiError, isEmpty } from '@yandex-infracloud-ui/libs';
import { Observable, of } from 'rxjs';
import { catchError, map } from 'rxjs/operators';

import { DeepPartial } from '../../../models';
import { prepareYpPaths, UsedYpObjectTypes, YpLocation, YpObjects, YpPaths } from '../../../models/api';
import {
   DeployTicket,
   DeployTicketConverter,
   ProjectItem,
   Release,
   ReleaseConverter,
   ReleaseRule,
   ReleaseRuleConverter,
   ReleaseRuleFormParams,
   ReleaseRuleType,
   ResourceType,
   TicketAction,
} from '../../../models/ui';
import { NodeFiltersParams } from '../../../models/ui/yp/view';
import {
   EAccessControlAction,
   EAccessControlPermission,
   EDeployTicketPatchSelectorType,
   EObjectType,
   TApprovalPolicy,
   TApprovalPolicySpec,
   TAutoCommitPolicy_EType,
   TDeployTicket,
   TDeployTicketControl_TActionOptions,
   TDeployTicketControl_TApproveOptions,
   TDeployTicketControl_TDisapproveOptions,
   TDeployTicketControl_TPatchSelector,
   TDeployTicketSpec_EDeployTicketSourceType,
   TDeployUnitApproval_TApprovalPayload,
   TMultiClusterReplicaSet,
   TProject,
   TRelease,
   TReleaseRule,
   TReleaseRuleSpec,
   TReplicaSet,
   TStage,
   TStageControl,
   TStageControl_TOverrideDeploymentStrategy,
   TStageControl_TOverrideDeploymentStrategy_TOverrideDeploySettings,
   TStageDraft,
   TStageSpec,
   TTimeInterval,
} from '../../../proto-typings';
import { noop } from '../../../utils';
import { getConfig } from '../../Config';

import {
   CheckObjectPermissionsParams,
   GetAvailableIdsByLoginParams,
   GetAvailableIdsByLoginRequest,
   SelectObjectResult,
   UpdateObjectParams,
   YpObjectServiceApiBase,
} from './ypObjectServiceApiBase';

export enum StageLocationAction {
   Approve = 'approve',
   Disapprove = 'disapprove',
}

interface HistoryReqOptions<T extends UsedYpObjectTypes> {
   type: T;
   id: string;
   uuid?: string;
   continuationToken?: string;
   limit?: number;
   paths?: string[] | YpPaths<T>;
   distinct?: boolean;
   interval?: TTimeInterval;
}

export interface SelectProjectReqOptions {
   substring?: string;
   tags?: string;
   paths?: string[] | YpPaths<EObjectType.OT_PROJECT>;
   continuationToken?: string;
   limit?: number;
   loadAll?: boolean;
}

export interface GetProjectReqOptions {
   ids: string[];
   paths?: string[] | YpPaths<EObjectType.OT_PROJECT>;
   fetchTimestamps?: boolean;
}

export interface GetPodsReqParams {
   location: YpLocation;
   podSetId?: string;
   limit: number;
   page: number;
   filter?: string;
}

export interface GetReplicaSetReqParams<T> {
   id: string;
   type: T;
   location?: YpLocation;
}

export interface SelectStagesReqOptions {
   substring?: string;
   continuationToken?: string;
   limit?: number;
   paths?: YpPaths<EObjectType.OT_STAGE> | string[];
   project?: string;
   fetchTimestamps?: boolean;
   tags?: string[];
}

export interface GetStagesReqOptions {
   objectIds: string[];
   paths?: YpPaths<EObjectType.OT_STAGE> | string[];
   fetchTimestamps?: boolean;
   tags?: string[];
}

export interface GetDeployTicketsReqOptions {
   stageId?: string;
   releaseId?: string;
   continuationToken?: string;
   limit?: number;
   filterQueryParams?: string[];
   loadAll?: boolean;
   paths?: YpPaths<EObjectType.OT_DEPLOY_TICKET> | string[];
   withOrder?: boolean;
}

type UpdateProjectReqOptions = Omit<UpdateObjectParams<null>, 'type'>;

// TODO: заменить старое api и пробросить конфиг
export interface DoStageLocationActionOptions {
   action: StageLocationAction;
   stageId: string;
   duId: string;
   duRevision: number | null;
   cluster: string;
}

export interface StageDeploySettingsOptions {
   stageId: string;
   duSettings: TStageControl_TOverrideDeploymentStrategy_TOverrideDeploySettings['deploy_settings'];
}

export interface OvverideStageDisruptionBudgetOptions {
   stageId: string;
   duId: string;
   duRevision: number;
   clusters: string[]; // multi -> https://a.yandex-team.ru/search?search=%22multi%22,%5Eyp.*,,arcadia,,5000&repo=arc
   value: number;
}

/**
 * класс, описывающий высокоуровневые методы взаимодействия с yp api
 * методы, которые используются непосредственно, обертки над абстрактными методами ypObjectServiceApiBase
 */
export class YpApi extends YpObjectServiceApiBase {
   public createReleaseRule(stageId: string, v: ReleaseRuleFormParams): Observable<unknown> {
      const releaseRule: DeepPartial<TReleaseRule> = {
         meta: { id: v.id.trim(), stage_id: stageId },
         spec: this.buildReleaseRuleSpec(v),
      };

      return this.createObject({ type: EObjectType.OT_RELEASE_RULE, data: releaseRule });
   }

   public createDraftAndTicket(id: string, stageId: string, spec: TStageSpec): Observable<unknown> {
      const draft: DeepPartial<TStageDraft> = {
         meta: {
            id,
            stage_id: stageId,
         },
         spec: {
            revision: 1,
            stage_spec: spec,
            stage_revision: spec.revision,
         },
      };

      const ticket: DeepPartial<TDeployTicket> = {
         meta: {
            id: `${id}-ticket`,
            stage_id: stageId,
         },
         spec: {
            source_type: TDeployTicketSpec_EDeployTicketSourceType.STAGE_DRAFT,
            stage_draft_id: id,
            stage_draft_revision: 1,
            description: spec.revision_info?.description || '',
         },
      };

      return this.createObjects([
         {
            type: EObjectType.OT_STAGE_DRAFT,
            data: draft,
         },
         {
            type: EObjectType.OT_DEPLOY_TICKET,
            data: ticket,
         },
      ]);
   }

   public getDrafts(draftIds: string[]): Observable<TStageDraft[]> {
      return this.getObjects({
         type: EObjectType.OT_STAGE_DRAFT,
         objectIds: draftIds,
      }).pipe(map(({ values: drafts }) => drafts as TStageDraft[]));
   }

   public getDeployTickets({
      stageId,
      releaseId,
      continuationToken,
      limit,
      paths,
      loadAll,
      filterQueryParams = [],
      withOrder = true,
   }: GetDeployTicketsReqOptions): Observable<SelectObjectResult<EObjectType.OT_DEPLOY_TICKET>> {
      const queryList: string[] = [];
      if (stageId) {
         queryList.push(`[/meta/stage_id]='${stageId}'`);
      }
      if (releaseId) {
         queryList.push(`[/spec/release_id]='${releaseId}'`);
      }
      if (filterQueryParams.length > 0) {
         queryList.push(...filterQueryParams);
      }

      const query = queryList.length > 0 ? queryList.join(' AND ') : undefined;

      return this.selectObjects({
         type: EObjectType.OT_DEPLOY_TICKET,
         query,
         limit,
         loadAll,
         continuationToken,
         paths,
         order_by: withOrder
            ? { 'expressions': [{ 'expression': '/meta/creation_time', 'descending': true }] }
            : undefined,
      });
   }

   public getDeployTicket(ticketId: string): Observable<DeployTicket> {
      return this.getObjects({
         type: EObjectType.OT_DEPLOY_TICKET,
         objectIds: [ticketId],
      }).pipe(
         map(({ values: items }) => (items as TDeployTicket[]).map(DeployTicketConverter.fromApi)),
         map(items => items[0]),
      );
   }

   public getReleaseRules(stageId: string): Observable<ReleaseRule[]> {
      return this.selectObjects({
         type: EObjectType.OT_RELEASE_RULE,
         query: `[/meta/stage_id]='${stageId}'`,
      }).pipe(map(items => (items.values as TReleaseRule[]).map(ReleaseRuleConverter.fromApi)));
   }

   public getReleaseRule(ruleId: string): Observable<ReleaseRule> {
      return this.getObjects({
         type: EObjectType.OT_RELEASE_RULE,
         objectIds: [ruleId],
      }).pipe(
         map(({ values: items }) => (items as TReleaseRule[]).map(ReleaseRuleConverter.fromApi)),
         map(items => items[0]),
      );
   }

   public updateReleaseRule(v: ReleaseRuleFormParams): Observable<unknown> {
      return this.updateObject({
         type: EObjectType.OT_RELEASE_RULE,
         id: v.id.trim(),
         paths: {
            '/spec': this.buildReleaseRuleSpec(v),
         },
         // paths: { ???
         //    [prepareYpPath(e => e.spec)]: this._buildReleaseRuleSpec(v),
         // },
      });
   }

   public getReleases(releaseIds: string[]): Observable<Release[]> {
      return this.getObjects({
         type: EObjectType.OT_RELEASE,
         objectIds: releaseIds,
      }).pipe(map(({ values: items }) => (items as TRelease[]).map(ReleaseConverter.fromApi)));
   }

   public getRelease(releaseId: string): Observable<Release> {
      return this.getReleases([releaseId]).pipe(map(items => items[0]));
   }

   public getPermissions(
      subrequests: CheckObjectPermissionsParams[],
   ): Observable<(EAccessControlAction | undefined)[]> {
      return this.checkObjectPermissions(subrequests);
   }

   public doTicketAction(
      action: TicketAction,
      ticketId: string,
      patchIds: string[],
      message: string,
      reason: string,
   ): Observable<unknown> {
      const patchSelector: Partial<TDeployTicketControl_TPatchSelector> =
         patchIds.length > 0
            ? { type: EDeployTicketPatchSelectorType.DTPST_PARTIAL, patch_ids: patchIds }
            : { type: EDeployTicketPatchSelectorType.DTPST_FULL };

      let options;

      if ([TicketAction.Approve, TicketAction.Disapprove].includes(action)) {
         options =
            action === TicketAction.Approve
               ? ({ message } as TDeployTicketControl_TApproveOptions)
               : ({ message } as TDeployTicketControl_TDisapproveOptions);
      } else {
         options = { message, patch_selector: patchSelector, reason } as TDeployTicketControl_TActionOptions;
      }

      return this.updateObject({
         id: ticketId,
         type: EObjectType.OT_DEPLOY_TICKET,
         paths: {
            [`/control/${action}`]: { options },
         },
      });
   }

   public removeRule(ruleId: string): Observable<unknown> {
      return this.deleteObject({ type: EObjectType.OT_RELEASE_RULE, id: ruleId });
   }

   public doStageLocationAction({
      action,
      stageId,
      duId,
      duRevision,
      cluster,
   }: DoStageLocationActionOptions): Observable<unknown> {
      const options: TDeployUnitApproval_TApprovalPayload = {
         deploy_unit: duId,
         cluster,
         revision: duRevision ?? 1,
      };
      return this.updateObject({
         id: stageId,
         type: EObjectType.OT_STAGE,
         paths: {
            [`/control/${action}`]: { options },
         },
      });
   }

   public overrideStageDisruptionBudget({
      stageId,
      duId,
      duRevision,
      clusters,
      value,
   }: OvverideStageDisruptionBudgetOptions): Observable<unknown> {
      const options: Partial<TStageControl_TOverrideDeploymentStrategy> = {
         max_unavailable: {
            deploy_unit_id: duId,
            clusters,
            revision: duRevision,
            value,
         },
      };

      const controlName: keyof TStageControl = 'override_deployment_strategy';
      return this.updateObject({
         id: stageId,
         type: EObjectType.OT_STAGE,
         paths: {
            [`/control/${controlName}`]: options,
         },
      });
   }

   public overrideStageDeploySettings({ stageId, duSettings }: StageDeploySettingsOptions): Observable<unknown> {
      const options: Partial<TStageControl_TOverrideDeploymentStrategy> = {
         deploy_settings: {
            deploy_settings: duSettings,
         },
      };

      const controlName: keyof TStageControl = 'override_deployment_strategy';
      return this.updateObject({
         id: stageId,
         type: EObjectType.OT_STAGE,
         paths: {
            [`/control/${controlName}`]: options,
         },
      });
   }

   public getFullStage(stageId: string): Observable<DeepPartial<TStage> | null> {
      return this.getObjects({
         type: EObjectType.OT_STAGE,
         objectIds: [stageId],
      }).pipe(map(({ values }) => values[0] ?? null));
   }

   public selectStages({
      substring = '',
      continuationToken,
      limit = 100,
      paths = e => [e.labels, e.meta, e.spec, e.status],
      project,
      fetchTimestamps = true,
      tags,
   }: SelectStagesReqOptions): Observable<SelectObjectResult<EObjectType.OT_STAGE>> {
      if (paths && !prepareYpPaths(paths).some(e => e.startsWith('/meta') || e === '')) {
         throw new Error('Path starting with "/meta" is expected');
      }

      const queryParts = [];

      // engine
      queryParts.push(`[/labels/deploy_engine]='${getConfig()!.getDeployEngine()}'`);

      // id
      if (substring) {
         queryParts.push(`is_substr('${substring}', [/meta/id])`);
      }

      // project
      if (project) {
         queryParts.push(`[/meta/project_id] = '${project}'`);
      }

      // tags
      if (tags) {
         for (const tag of tags) {
            queryParts.push(`list_contains([/labels/tags], '${tag}')`);
         }
      }

      return this.selectObjects({
         type: EObjectType.OT_STAGE,
         limit,
         continuationToken,
         paths,
         query: queryParts.join(' AND '),
         fetchTimestamps,
      });
   }

   public getStages({
      objectIds,
      paths = [''],
      fetchTimestamps = false,
   }: GetStagesReqOptions): Observable<DeepPartial<TStage>[]> {
      return this.getObjects({
         type: EObjectType.OT_STAGE,
         objectIds,
         paths,
         fetchTimestamps,
      }).pipe(map(({ values }) => values));
   }

   // TODO: @nikolaichev: перенести на уровень выше
   // Используется только для E2E тестов
   public getRawStage(stageId: string): Observable<TStage | null> {
      if (isEmpty(stageId)) {
         return of(null);
      }

      return this.getObjects({
         type: EObjectType.OT_STAGE,
         objectIds: [stageId!],
      }).pipe(
         map(({ values: stages }) => (stages.length > 0 ? (stages[0] as TStage) : null)),
         catchError(err => {
            // обрабатываем ситуацию, когда ищем несуществующий стейдж
            if (err.data?.ytError?.code === 100002) {
               return of(null);
            }
            throw err;
         }),
      );
   }

   public getStage(
      stageId: string,
      paths?: YpPaths<EObjectType.OT_STAGE> | string[],
   ): Observable<DeepPartial<TStage> | null> {
      if (isEmpty(stageId)) {
         return of(null);
      }

      return this.getStages({ objectIds: [stageId], paths }).pipe(
         map(stages => (stages && stages.length ? stages[0] : null)),
      );
   }

   public isStageChanged(stageId: string, previousRevision: number | undefined): Observable<boolean> {
      return this.getStage(stageId, ['/spec/revision']).pipe(
         map(actualStage => previousRevision !== actualStage?.spec?.revision),
      );
   }

   public createStage(stage: TStage): Observable<unknown> {
      return this.createObject({ type: EObjectType.OT_STAGE, data: stage });
   }

   public updateStage(
      newStage: TStage,
      updateProject: boolean,
      updateAccountId: boolean,
      updateTags: boolean,
      updateInfra: boolean,
      updatePatchersAutoupdateRevision: boolean,
   ): Observable<unknown> {
      return this.updateObject({
         id: newStage.meta!.id!,
         type: EObjectType.OT_STAGE,
         paths: {
            '/spec': newStage.spec,
            ...(updateProject ? { '/meta/project_id': newStage.meta?.project_id } : {}),
            ...(updateAccountId ? { '/meta/account_id': newStage.meta?.account_id } : {}),
            ...(updateTags ? { '/labels/tags': newStage.labels?.tags || [] } : {}),
            ...(updateInfra ? { '/labels/infra_service': newStage.labels?.infra_service } : {}),
            ...(updateInfra ? { '/labels/infra_service_name': newStage.labels?.infra_service_name } : {}),
            ...(updateInfra ? { '/labels/infra_environment': newStage.labels?.infra_environment } : {}),
            ...(updateInfra ? { '/labels/infra_environment_name': newStage.labels?.infra_environment_name } : {}),
            // https://st.yandex-team.ru/DEPLOY-5195#62068a4d19c28f389b25b928
            ...(updatePatchersAutoupdateRevision
               ? { '/labels/du_patchers_autoupdate_revision': newStage.labels?.du_patchers_autoupdate_revision }
               : {}),
         },
      });
   }

   public updateStageDisabledClusters(
      stageId: string,
      disabledClusters: string[],
      existDeployLabel: boolean,
   ): Observable<unknown> {
      let updateData: Record<string, any> = {};
      if (disabledClusters.length > 0) {
         updateData = existDeployLabel
            ? {
                 '/labels/deploy/disabled_clusters': disabledClusters,
              }
            : {
                 '/labels/deploy': { disabled_clusters: disabledClusters },
              };
      } else {
         updateData = existDeployLabel
            ? {
                 '/labels/deploy/disabled_clusters': undefined,
              }
            : {
                 '/labels/deploy': undefined,
              };
      }
      return this.updateObject({
         id: stageId,
         type: EObjectType.OT_STAGE,
         paths: updateData,
      });
   }

   public deleteStage(stageId: string): Observable<unknown> {
      return this.deleteObject({
         type: EObjectType.OT_STAGE,
         id: stageId,
      });
   }

   // TODO: @nikolaichev: перенести на уровень выше
   // Используется только для E2E тестов
   public getRawProject(projectId: string): Observable<TProject | null> {
      if (isEmpty(projectId)) {
         return of(null);
      }

      return this.getObjects({
         type: EObjectType.OT_PROJECT,
         objectIds: [projectId!],
      }).pipe(
         map(({ values: projects }) => (projects.length > 0 ? (projects[0] as TProject) : null)),
         catchError(err => {
            // обрабатываем ситуацию, когда ищем несуществующий проект
            if (err.data?.ytError?.code === 100002) {
               return of(null);
            }
            throw err;
         }),
      );
   }

   public getProject(projectId: string | undefined): Observable<ProjectItem | null> {
      if (isEmpty(projectId)) {
         return of(null);
      }

      return this.getObjects({
         type: EObjectType.OT_PROJECT,
         objectIds: [projectId as string],
         paths: p => [p.meta.id, p.spec.account_id],
      }).pipe(
         map(({ values: projects }) =>
            projects.length > 0
               ? projects.map(p => ({
                    id: p.meta!.id!,
                    accountId: p.spec!.account_id!,
                 }))[0]
               : null,
         ),
      );
   }

   public getProjects({
      ids,
      paths = [''],
      fetchTimestamps = false,
   }: GetProjectReqOptions): Observable<DeepPartial<TProject>[]> {
      return this.getObjects({
         type: EObjectType.OT_PROJECT,
         objectIds: ids,
         paths,
         fetchTimestamps,
      }).pipe(map(({ values }) => values));
   }

   public createProject(data: DeepPartial<TProject>): Observable<unknown> {
      return this.createObject({
         type: EObjectType.OT_PROJECT,
         data,
      });
   }

   public updateProject({ id, paths }: UpdateProjectReqOptions): Observable<unknown> {
      return this.updateObject({
         type: EObjectType.OT_PROJECT,
         id,
         paths,
      });
   }

   public selectProjects({
      substring,
      continuationToken,
      limit = 100,
      paths = e => [e.meta.id],
      loadAll = false,
      tags,
   }: SelectProjectReqOptions): Observable<SelectObjectResult<EObjectType.OT_PROJECT>> {
      const queryParts = [];

      // id
      if (substring) {
         queryParts.push(`is_substr('${substring}', [/meta/id])`);
      }

      // tags
      if (tags) {
         for (const tag of tags) {
            queryParts.push(`list_contains([/labels/tags], '${tag}')`);
         }
      }

      return this.selectObjects({
         type: EObjectType.OT_PROJECT,
         query: queryParts.join(' AND '),
         paths,
         continuationToken,
         limit,
         loadAll,
      });
   }

   public deleteProject(projectId: string): Observable<unknown> {
      return this.deleteObject({
         type: EObjectType.OT_PROJECT,
         id: projectId,
      });
   }

   public getHistory<T extends UsedYpObjectTypes>(options: HistoryReqOptions<T>, location?: YpLocation) {
      return this.selectObjectHistory(
         {
            ...options,
         },
         location,
      );
   }

   public fetchApprovalPolicy(stageId: string): Observable<TApprovalPolicy | undefined> {
      return this.selectObjects({
         type: EObjectType.OT_APPROVAL_POLICY,
         limit: 1,
         query: `[/meta/stage_id]='${stageId}'`,
      }).pipe(map(resp => resp.values[0] as TApprovalPolicy | undefined));
   }

   /**
    * Id политики апрувов всегда совпадает с id стейджа
    */
   public fetchApprovalPolicies(stageIds: string[]): Observable<TApprovalPolicy[]> {
      if (stageIds.length === 0) {
         return of([]);
      }
      let query: string;
      if (stageIds.length === 1) {
         query = `[/meta/id]="${stageIds[0]}"`;
      } else {
         query = `[/meta/id] IN (${stageIds.map(id => `"${id}"`).join(', ')})`;
      }
      return this.selectObjects({
         type: EObjectType.OT_APPROVAL_POLICY,
         limit: stageIds.length,
         query,
      }).pipe(map(result => result.values as TApprovalPolicy[])); // запрашиваем все поля
   }

   public createApprovalPolicy(data: DeepPartial<TApprovalPolicy>): Observable<unknown> {
      return this.createObject({
         type: EObjectType.OT_APPROVAL_POLICY,
         data,
      });
   }

   public updateApprovalPolicy(id: string, spec: DeepPartial<TApprovalPolicySpec>): Observable<unknown> {
      return this.updateObject({
         id,
         type: EObjectType.OT_APPROVAL_POLICY,
         paths: {
            '/spec': spec,
         },
      });
   }

   public deleteApprovalPolicy(id: string): Observable<unknown> {
      return this.deleteObject({
         type: EObjectType.OT_APPROVAL_POLICY,
         id,
      });
   }

   public getPods({ podSetId, location, limit, page, filter }: GetPodsReqParams) {
      const offset = (page - 1) * limit;
      const query = [podSetId && `[/meta/pod_set_id]='${podSetId}'`, filter && `(${filter})`]
         .filter(Boolean)
         .join(' AND ');

      return this.selectObjects({
         type: EObjectType.OT_POD,
         limit: limit + 1, // TODO: переделать на пагинацию с токеном
         offset,
         query,
         location,
         headers: {
            'X-YT-Response-Format-Options': '{encode_utf8=%true}',
         },
      });
   }

   public getReplicaSet<T extends EObjectType.OT_REPLICA_SET | EObjectType.OT_MULTI_CLUSTER_REPLICA_SET>({
      id,
      location,
      type,
   }: GetReplicaSetReqParams<T>): Observable<DeepPartial<YpObjects[T]> | null> {
      return this.getObjects({
         type,
         ...(type === EObjectType.OT_REPLICA_SET ? { location } : {}),
         objectIds: [id],
      }).pipe(map(({ values }) => values[0] ?? null));
   }

   // TODO: метод выпилится.
   // Будет вмёржен в develop в этой ветке. После чего в ветке feature/getPods-continuation-token
   // в процессе рефакторинга стейта будет унифицирован с методом выше
   public getClusterReplicaSet(location: YpLocation, replicaSetId: string): Observable<DeepPartial<TReplicaSet>> {
      return this.getObjects({
         location,
         type: EObjectType.OT_REPLICA_SET,
         objectIds: [replicaSetId],
      }).pipe(map(({ values }) => values[0]));
   }

   public getMulticlusterReplicaSet(replicaSetId: string): Observable<DeepPartial<TMultiClusterReplicaSet>> {
      return this.getObjects({
         location: YpLocation.XDC,
         type: EObjectType.OT_MULTI_CLUSTER_REPLICA_SET,
         objectIds: [replicaSetId],
      }).pipe(map(({ values }) => values[0]));
   }

   public getAvailableNetworkProjects(login: string) {
      return this.getAvailableIdsByLogin({
         requests: [
            {
               objectType: EObjectType.OT_NETWORK_PROJECT,
               login,
               permission: EAccessControlPermission.ACA_USE,
               limit: 1000,
            },
         ],
      }).pipe(map(rsp => rsp[0]));
   }

   public getAvailableIds({ requests }: { requests: GetAvailableIdsByLoginParams }) {
      const engine = `[/labels/deploy_engine]='${getConfig()!.getDeployEngine()}'`;
      // const inProject = ` AND [/meta/project_id] = '${project}'`; // после включения в yp
      return this.getAvailableIdsByLogin({
         requests: requests.map(e => {
            if (e.objectType === EObjectType.OT_STAGE) {
               e.query = engine;
            }
            return e;
         }),
      });
   }

   public createRelease(data: DeepPartial<TRelease>): Observable<unknown> {
      return this.createObject({
         type: EObjectType.OT_RELEASE,
         data,
      });
   }

   public deleteRelease(releaseId: string): Observable<unknown> {
      return this.deleteObject({
         type: EObjectType.OT_RELEASE,
         id: releaseId,
      });
   }

   public getYPNode(nodeId: string, location: YpLocation) {
      return this.getObjects({
         type: EObjectType.OT_NODE,
         objectIds: [nodeId],
         location,
         paths: e => [
            e.meta,
            e.labels,
            e.spec,
            e.status.epoch_id,
            e.status.last_seen_time,
            e.status.last_handshake_time,
            e.status.last_handshake_master_address,
            e.status.heartbeat_sequence_number,
            e.status.agent_address,
            e.status.agent_version,
            e.status.hfsm,
            e.status.maintenance,
            e.status.alerts,
            e.status.unknown_pod_ids,
            e.status.unknown_persistent_volume_ids,
         ],
      }).pipe(map(({ values }) => values[0]));
   }

   public getYPNodes(
      location: YpLocation,
      { nodeId, state, maintenance, segments, query }: NodeFiltersParams,
      continuationToken?: string,
   ) {
      const limit = 50;
      const queryParts = [];
      if (nodeId) queryParts.push(`is_substr('${nodeId}', [/meta/id])`);
      if (state && state.length) {
         const statesQuery = state.map((stateItem: string) => `[/status/hfsm/state]="${stateItem}"`).join(' OR ');

         queryParts.push(`(${statesQuery})`);
      }
      if (maintenance && maintenance.length) {
         const maintenanceStatesQuery = maintenance
            .map((maintenanceItem: string) => `[/status/maintenance/state]="${maintenanceItem}"`)
            .join(' OR ');

         queryParts.push(`(${maintenanceStatesQuery})`);
      }
      if (segments && segments.length) {
         const segmentsQuery = segments.map((stateItem: string) => `[/labels/segment]="${stateItem}"`).join(' OR ');

         queryParts.push(`(${segmentsQuery})`);
      }
      if (query) queryParts.push(query);
      return this.selectObjects({
         type: EObjectType.OT_NODE,
         location,
         continuationToken,
         query: queryParts.join(' AND '),
         limit,
         paths: e => [e.meta, e.status.hfsm, e.labels, e.status.maintenance, e.status.last_seen_time],
      }).pipe(
         map(items => ({
            items: items.values,
            continuationToken: items.continuationToken,
            lastPage: items.values.length < limit,
         })),
      );
   }

   public getYPNodesSegments(location: YpLocation) {
      return this.selectObjects({
         type: EObjectType.OT_NODE_SEGMENT,
         location,
         paths: e => [e.meta],
      }).pipe(map(res => res.values));
   }

   public getNodeResources(nodeId: string, location: YpLocation) {
      return this.selectObjects({
         type: EObjectType.OT_RESOURCE,
         location,
         query: `[/meta/node_id]="${nodeId}"`,
      }).pipe(map(response => response.values));
   }

   public getAccount(abcId: string, location: YpLocation) {
      return this.getObjects({
         type: EObjectType.OT_ACCOUNT,
         objectIds: [abcId],
         location,
      }).pipe(map(response => response.values[0]));
   }

   public getPodAnnotated(podId: string, location: YpLocation) {
      return this.getObjects({
         type: EObjectType.OT_POD,
         objectIds: [podId],
         location,
         paths: e => [e.status.agent.current_spec_timestamp, e.status.pod_dynamic_attributes.annotations],
         annotateTypes: true,
      }).pipe(map(({ values, timestamp }) => ({ pod: values[0], timestamp })));
   }

   public getPod(podId: string, location: YpLocation) {
      return this.getObjects({
         type: EObjectType.OT_POD,
         objectIds: [podId],
         location,
         paths: e => [
            e.meta,
            e.meta.effective_account_id,
            e.labels,
            e.spec.account_id,
            e.spec.capabilities,
            e.spec.disk_volume_claims,
            e.spec.disk_volume_requests,
            e.spec.dynamic_attributes,
            e.spec.dynamic_resources,
            e.spec.enable_scheduling,
            e.spec.gpu_requests,
            e.spec.host_devices,
            e.spec.host_infra,
            e.spec.host_name_kind,
            e.spec.ip6_address_requests,
            e.spec.ip6_subnet_requests,
            e.spec.iss,
            e.spec.node_filter,
            e.spec.node_id,
            e.spec.out_of_memory_policy,
            e.spec.pod_agent_payload.meta,
            e.spec.pod_agent_payload.spec,
            e.spec.resource_cache,
            e.spec.resource_requests,
            e.spec.scheduling,
            e.spec.sysctl_properties,
            e.spec.virtual_service_options,
            e.status.agent.current_spec_applied,
            e.status.agent.current_spec_timestamp,
            e.status.agent.execution_error,
            e.status.agent.failed_install_attempt_spec_timestamp,
            e.status.agent.iss,
            e.status.agent.last_heartbeat_time,
            e.status.agent.pod_agent_payload,
            e.status.agent.state,
            e.status.agent.validation_failures,
            e.status.agent_spec_timestamp,
            e.status.aggregated_resource_usage_snapshot,
            e.status.cpu_allocation,
            e.status.disk_volume_allocations,
            e.status.disk_volume_mounts,
            e.status.dns,
            e.status.dynamic_resources,
            e.status.eviction,
            e.status.generation_number,
            e.status.gpu_allocations,
            e.status.ip6_address_allocations,
            e.status.ip6_subnet_allocations,
            e.status.latest_resource_usage_snapshot,
            e.status.maintenance,
            e.status.master_spec_timestamp,
            e.status.node_alerts,
            e.status.pod_dynamic_attributes.labels,
            e.status.scheduled_resource_allocations,
            e.status.scheduling,
         ],
      }).pipe(map(({ values, timestamp }) => ({ pod: values[0], timestamp })));
   }

   public requestPodEviction(podId: string, timestamp: number, message: string, location: YpLocation) {
      return this.updateObject({
         id: podId,
         type: EObjectType.OT_POD,
         paths: {
            '/control/request_eviction': {
               message,
               validate_disruption_budget: false,
            },
         },
         prerequisites: [
            { path: '/status/eviction', timestamp },
            { path: '/status/scheduling', timestamp },
         ],
         location,
      });
   }

   public abortPodEviction(podId: string, timestamp: number, message: string, location: YpLocation) {
      return this.updateObject({
         id: podId,
         type: EObjectType.OT_POD,
         paths: {
            '/control/abort_eviction': {
               message,
            },
         },
         prerequisites: [
            { path: '/status/eviction', timestamp },
            { path: '/status/scheduling', timestamp },
         ],
         location,
      });
   }

   public acknowledgePodEviction(podId: string, timestamp: number, message: string, location: YpLocation) {
      return this.updateObject({
         id: podId,
         type: EObjectType.OT_POD,
         paths: {
            '/control/acknowledge_eviction': {
               message,
            },
         },
         prerequisites: [
            { path: '/status/eviction', timestamp },
            { path: '/status/scheduling', timestamp },
         ],
         location,
      });
   }

   public getNodePods(nodeId: string, location: YpLocation) {
      return this.selectObjects({
         type: EObjectType.OT_POD,
         location,
         loadAll: true,
         query: `[/spec/node_id]="${nodeId}"`,
         paths: e => [
            e.meta,
            e.labels,
            e.spec.account_id,
            e.spec.capabilities,
            e.spec.disk_volume_claims,
            e.spec.disk_volume_requests,
            e.spec.dynamic_attributes,
            e.spec.dynamic_resources,
            e.spec.enable_scheduling,
            e.spec.gpu_requests,
            e.spec.host_devices,
            e.spec.host_infra,
            e.spec.host_name_kind,
            e.spec.ip6_address_requests,
            e.spec.ip6_subnet_requests,
            e.spec.iss,
            e.spec.node_filter,
            e.spec.node_id,
            e.spec.out_of_memory_policy,
            e.spec.pod_agent_payload.meta,
            e.spec.pod_agent_payload.spec,
            e.spec.resource_cache,
            e.spec.resource_requests,
            e.spec.scheduling,
            e.spec.sysctl_properties,
            e.spec.virtual_service_options,
            e.status.agent.current_spec_applied,
            e.status.agent.current_spec_timestamp,
            e.status.agent.execution_error,
            e.status.agent.failed_install_attempt_spec_timestamp,
            e.status.agent.iss,
            e.status.agent.last_heartbeat_time,
            e.status.agent.pod_agent_payload,
            e.status.agent.state,
            e.status.agent.validation_failures,
            e.status.agent_spec_timestamp,
            e.status.aggregated_resource_usage_snapshot,
            e.status.cpu_allocation,
            e.status.disk_volume_allocations,
            e.status.disk_volume_mounts,
            e.status.dns,
            e.status.dynamic_resources,
            e.status.eviction,
            e.status.generation_number,
            e.status.gpu_allocations,
            e.status.ip6_address_allocations,
            e.status.ip6_subnet_allocations,
            e.status.latest_resource_usage_snapshot,
            e.status.maintenance,
            e.status.master_spec_timestamp,
            e.status.node_alerts,
            e.status.pod_dynamic_attributes,
            e.status.scheduled_resource_allocations,
            e.status.scheduling,
         ],
      }).pipe(map(({ values }) => values));
   }

   public getPodSets(location: YpLocation, query?: string, paths?: string[], continuationToken?: string, limit = 50) {
      return this.selectObjects({
         location,
         type: EObjectType.OT_POD_SET,
         continuationToken,
         limit,
         query,
         paths:
            paths ??
            (e => [
               e.meta.id,
               e.labels,
               e.spec.account_id,
               e.spec.node_segment_id,
               e.spec.node_filter,
               e.spec.antiaffinity_constraints,
            ]),
      }).pipe(
         map(items => ({
            values: items.values,
            continuationToken: items.continuationToken,
            lastPage: items.values.length < limit,
         })),
      );
   }

   public getPodSet(podSetId: string, location: YpLocation) {
      return this.getObjects({
         location,
         objectIds: [podSetId],
         type: EObjectType.OT_POD_SET,
      }).pipe(map(({ values }) => values[0]));
   }

   protected getCSRF(): string {
      return '';
   }

   protected handleError(resp: Response, error: IApiError): void {
      noop();
   }

   private buildReleaseRuleSpec(formParams: ReleaseRuleFormParams): DeepPartial<TReleaseRuleSpec> {
      const result: DeepPartial<TReleaseRuleSpec> = {
         auto_commit_policy: {
            type: formParams.autocommit ? TAutoCommitPolicy_EType.MAINTAIN_ACTIVE_TRUNK : TAutoCommitPolicy_EType.NONE,
         },
         description: formParams.description,
         patches: {},
      };

      switch (formParams.type) {
         case ReleaseRuleType.Sandbox: {
            const sandboxParams = formParams.sandbox!;

            result.sandbox = {
               attributes: sandboxParams.attributes
                  .filter(v => !isEmpty(v.key.trim()) && !isEmpty(v.value.trim()))
                  .reduce((acc, item) => {
                     acc[item.key.trim()] = item.value;

                     return acc;
                  }, {} as Record<string, string>),
               release_types: formParams.releaseTypes,
               resource_types: sandboxParams.resources.map(v => v.trim()),
               task_type: sandboxParams.taskType.trim(),
            };

            for (const patch of sandboxParams.patches) {
               if (patch.type === ResourceType.DynamicResource) {
                  result.patches![patch.id] = {
                     sandbox: {
                        sandbox_resource_type: patch.resourceType.trim(),
                        dynamic: {
                           dynamic_resource_id: patch.ref,
                           deploy_group_mark: patch.deployGroupMark,
                        },
                     },
                  };
               } else {
                  result.patches![patch.id] = {
                     sandbox: {
                        sandbox_resource_type: patch.resourceType.trim(),
                        static: {
                           deploy_unit_id: patch.deployUnitId,
                           [patch.type === ResourceType.Layer ? 'layer_ref' : 'static_resource_ref']: patch.ref,
                        },
                     },
                  };
               }
            }
            break;
         }

         case ReleaseRuleType.Docker: {
            const dockerParams = formParams.docker!;

            result.docker = {
               image_name: dockerParams.image.trim(),
               release_types: formParams.releaseTypes,
            };

            for (const patch of dockerParams.patches) {
               result.patches![patch.id] = {
                  docker: {
                     docker_image_ref: {
                        box_id: patch.boxId,
                        deploy_unit_id: patch.deployUnitId,
                     },
                  },
               };
            }

            break;
         }
      }

      return result;
   }

   public getUserAllowedAccounts(login: string, location: YpLocation) {
      const abcServiceRegex = /abc:service:([0-9]+)/;
      const requests: GetAvailableIdsByLoginRequest[] = [
         {
            objectType: EObjectType.OT_ACCOUNT,
            permission: EAccessControlPermission.ACA_USE,
            login,
            path: '/',
         },
      ];

      return this.getAvailableIdsByLogin({ requests }, location).pipe(
         map(value => {
            const filteredServices = value.flat().filter(x => x === 'tmp' || abcServiceRegex.test(x));

            const servicesSet = new Set<string>(filteredServices);
            return Array.from(servicesSet);
         }),
      );
   }

   public getAccountsByIds(location: YpLocation, accountIds: string[]) {
      return this.getObjects({
         objectIds: accountIds,
         location,
         type: EObjectType.OT_ACCOUNT,
         paths: e => [e.meta, e.status.resource_usage, e.spec.resource_limits],
      }).pipe(map(response => response.values));
   }

   public getReplicaSets(location: YpLocation, filter?: string, continuationToken?: string, limit?: number) {
      return this.selectObjects({
         type: EObjectType.OT_REPLICA_SET,
         location,
         query: filter,
         limit,
         continuationToken,
      });
   }

   public getMulticlusterReplicaSets(filter?: string, continuationToken?: string, limit?: number) {
      return this.selectObjects({
         type: EObjectType.OT_MULTI_CLUSTER_REPLICA_SET,
         location: YpLocation.XDC,
         query: filter,
         limit,
         continuationToken,
      });
   }
}
