import { BaseApi } from '@yandex-infracloud-ui/libs';
import { from, Observable } from 'rxjs';
import { catchError, map } from 'rxjs/operators';

import { DeepPartial } from '../../../models';
import {
   ApiErrorItemWithData,
   ApiRequest,
   ApiServiceName,
   prepareApiError,
   RequestMetaData,
} from '../../../models/api';
import {
   NetworkYpErrorData,
   prepareYpPaths,
   saveYpPaths,
   updateYpObject,
   UsedYpObjects,
   UsedYpObjectTypes,
   YpApiActions,
   YpApiParams,
   YpAttributeList,
   YpHistoryEvent,
   YpLocation,
   YpObjects,
   YpPaths,
   YpRspSelectObjectHistory,
   YpSelectObjectHistory,
} from '../../../models/api/services/yp';
import {
   EAccessControlAction,
   EAccessControlPermission,
   EPayloadFormat,
   TAttributeList,
   TAttributeTimestampPrerequisite,
   TObjectOrderBy,
   TReqGetObjects,
   TReqGetUserAccessAllowedTo,
   TReqSelectObjectHistory,
   TReqSelectObjects,
   TReqUpdateObject,
   TRspCreateObject,
   TRspCreateObjects,
   TRspRemoveObject,
   TRspUpdateObject,
   TSelectObjectsOptions,
   TTimeInterval,
} from '../../../proto-typings';
import { parseYpTimestamp } from '../../../utils';
import { getApiRequestParams } from '../requestParams';

const DEFAULT_YP_LIMIT = 500;

interface YpRequestAdditionalParams {
   location?: YpLocation;
}

interface RequestYpParams {
   location?: YpLocation;
   headers?: Record<string, string>;
}

interface GetObjectsParams<T extends UsedYpObjectTypes> extends RequestYpParams {
   objectIds: string[];
   type: T;
   paths?: string[] | YpPaths<T>;
   fetchTimestamps?: boolean;
   annotateTypes?: boolean;
}

interface SelectObjectsParams<T extends UsedYpObjectTypes> extends RequestYpParams {
   limit?: number;
   offset?: number;
   loadAll?: boolean;
   paths?: string[] | YpPaths<T>;
   query?: string;
   type: T;
   continuationToken?: string;
   fetchTimestamps?: boolean;
   order_by?: TObjectOrderBy;
}

export type SelectObjectResult<T extends UsedYpObjectTypes> = {
   values: DeepPartial<YpObjects[T]>[];
   continuationToken?: string;
};

export interface UpdateObjectParams<T> extends RequestYpParams {
   id: string;
   type: T;
   paths: Record<string, any>;
   prerequisites?: TAttributeTimestampPrerequisite[];
}

interface RemoveObjectParams<T> extends RequestYpParams {
   id: string;
   type: T;
}

interface YpObjectParams<T extends UsedYpObjectTypes> {
   type: T;
   data: DeepPartial<YpObjects[T]>;
}

type CreateObjectParams<T extends UsedYpObjectTypes> = YpObjectParams<T>;

type CreateObjectsParams = YpObjectParams<UsedYpObjectTypes>[];

export interface SelectObjectHistoryParams<T extends UsedYpObjectTypes> {
   type: T;
   id: string;
   uuid?: string;
   continuationToken?: string;
   limit?: number;
   paths?: string[] | YpPaths<T>;
   distinct?: boolean;
   interval?: TTimeInterval;
}

export interface CheckObjectPermissionsParams {
   type: UsedYpObjectTypes;
   id: string;
   login: string;
   permission?: EAccessControlPermission;
   path?: string;
}

export interface GetAvailableIdsByLoginRequest {
   continuationToken?: string;
   limit?: number;
   objectType: UsedYpObjectTypes;
   login?: string;
   query?: string;
   permission?: EAccessControlPermission;
   path?: string;
}

export type GetAvailableIdsByLoginParams = GetAvailableIdsByLoginRequest[];

interface LoadByPageParams<T> {
   continuationToken?: string;
   prevValues?: DeepPartial<T>[];
   limit: number;

   loader(token?: string): Promise<LoadByPageResult<T>>;
}

interface LoadByPageResult<T> {
   values: DeepPartial<T>[];
   continuationToken?: string;
}

interface YPObjectResult<T extends UsedYpObjectTypes> {
   values: DeepPartial<YpObjects[T]>[];
   timestamp?: number;
}

/**
 * Обёртка над нативным YP API, упрощает форму аргументов для запросов и парсит ответ(в частности, учитывает paths)
 */
export abstract class YpObjectServiceApiBase extends BaseApi {
   constructor(private endpoints: Record<YpLocation, string | null>, private devMode: boolean = false) {
      super(null);
   }

   protected createObject<T extends UsedYpObjectTypes>({
      type,
      data,
   }: CreateObjectParams<T>): Observable<DeepPartial<TRspCreateObject>> {
      return this.apiRequest({
         action: YpApiActions.CreateObject,
         data: {
            object_type: type,
            attributes_payload: {
               yson: data as any,
            },
         },
      });
   }

   protected createObjects(subrequests: CreateObjectsParams): Observable<DeepPartial<TRspCreateObjects>> {
      return this.apiRequest({
         action: YpApiActions.CreateObjects,
         data: {
            subrequests: subrequests.map(item => ({
               object_type: item.type,
               attributes_payload: {
                  yson: item.data as any,
               },
            })),
         },
      });
   }

   public static annotateHeaders = {
      'X-YT-Response-Format-Options': '{annotate_with_types=%true;stringify=%true}',
   };

   protected getObjects<T extends UsedYpObjectTypes>({
      type,
      objectIds,
      paths = [''],
      location,
      fetchTimestamps = false,
      annotateTypes = false,
   }: GetObjectsParams<T>): Observable<YPObjectResult<T>> {
      saveYpPaths(YpApiActions.GetObjects, type, paths);

      const preparedPaths = prepareYpPaths(paths);

      const input: DeepPartial<TReqGetObjects> = {
         object_type: type,
         selector: {
            paths: preparedPaths,
         },
         subrequests: objectIds.map(id => ({ object_id: id })),
         format: EPayloadFormat.PF_YSON,
         options: {
            fetch_timestamps: fetchTimestamps, // TODO: поддержать исключение несхематизированных атрибутов
         },
      };

      const headers: Record<string, string> = annotateTypes ? YpObjectServiceApiBase.annotateHeaders : {};

      return this.apiRequest({ action: YpApiActions.GetObjects, data: input, location, headers }).pipe(
         map(resp => {
            const result: YPObjectResult<T> = {
               values: (resp.subresponses ?? []).map(item =>
                  this.parseYpResponse<YpObjects[T]>(item.result!, preparedPaths),
               ),
               timestamp: resp.timestamp,
            };

            return result;
         }),
      );
   }

   protected deleteObject<T extends UsedYpObjectTypes>({
      type,
      id,
   }: RemoveObjectParams<T>): Observable<DeepPartial<TRspRemoveObject>> {
      return this.apiRequest({
         action: YpApiActions.RemoveObject,
         data: {
            object_type: type,
            object_id: id,
         },
      });
   }

   protected selectObjectHistory<T extends UsedYpObjectTypes>(
      { type, id, uuid, continuationToken, limit, paths = [''], distinct, interval }: SelectObjectHistoryParams<T>,
      location: YpLocation = YpLocation.XDC,
   ): Observable<YpSelectObjectHistory<YpObjects[T]>> {
      saveYpPaths(YpApiActions.SelectObjectHistory, type, paths);

      const input: DeepPartial<TReqSelectObjectHistory> = {
         object_type: type,
         object_id: id,
         selector: {
            paths: prepareYpPaths(paths),
         },
         format: EPayloadFormat.PF_YSON,
         options: {
            uuid,
            limit: limit ?? 100,
            continuation_token: continuationToken,
            descending_time_order: true,
            interval,
            distinct: distinct ?? true,
         },
      };

      return this.apiRequest({ action: YpApiActions.SelectObjectHistory, data: input, location }).pipe(
         map(rawResponse => {
            const response = rawResponse as YpRspSelectObjectHistory;
            const events = response.events ?? [];
            for (const event of events) {
               (event as YpHistoryEvent<YpObjects[T]>).results = this.parseYpResponse<YpObjects[T]>(
                  event.results!,
                  prepareYpPaths(paths),
               );
            }
            return response as YpSelectObjectHistory<YpObjects[T]>;
         }),
      );
   }

   // TODO: убрать DeepPartial при отсутствии paths
   protected selectObjects<T extends UsedYpObjectTypes>({
      type,
      limit = DEFAULT_YP_LIMIT,
      offset,
      query = '',
      loadAll = false,
      paths = [''],
      continuationToken,
      fetchTimestamps = false,
      location,
      headers,
      order_by,
   }: SelectObjectsParams<T>): Observable<SelectObjectResult<T>> {
      saveYpPaths(YpApiActions.SelectObjects, type, paths);
      const input: DeepPartial<TReqSelectObjects> = {
         object_type: type,
         selector: { paths: prepareYpPaths(paths) },
         ...(query ? { filter: { query } } : {}),
         format: EPayloadFormat.PF_YSON, // yson format for timestamps
         options: {
            limit,
            ...(offset ? { offset } : {}),
            continuation_token: continuationToken,
            fetch_timestamps: fetchTimestamps,
         },
         order_by,
      };

      if (!loadAll) {
         return this.apiRequest({
            action: YpApiActions.SelectObjects,
            data: input,
            location,
            headers,
         }).pipe(
            map(resp => ({
               values: (resp.results ?? []).map(result =>
                  this.parseYpResponse<YpObjects[T]>(result, prepareYpPaths(paths)),
               ),
               continuationToken: resp.continuation_token,
            })),
         );
      }

      // для исключения тяжелых запросов, с целью разбить их на мелкие
      const requestLimit = Math.min(limit, DEFAULT_YP_LIMIT);

      // loadAll используется, когда нам не нужно отдельно манипулировать токенами
      const valuesPromise = this.loadByPage<YpObjects[T]>({
         limit,
         loader: token => {
            const options: Partial<TSelectObjectsOptions> = {};
            if (token) {
               options.continuation_token = token;
            }
            return new Promise((resolve, reject) => {
               this.apiRequest({
                  action: YpApiActions.SelectObjects,
                  data: {
                     ...input,
                     options: {
                        ...input.options,
                        continuation_token: token,
                        limit: requestLimit,
                     },
                  },
                  location,
                  headers,
               }).subscribe(resp => {
                  const values = (resp.results ?? []).map(result =>
                     this.parseYpResponse<YpObjects[T]>(result, prepareYpPaths(paths)),
                  );
                  resolve({
                     values,
                     continuationToken: resp.continuation_token,
                  });
               }, reject);
            });
         },
      });

      return from(valuesPromise);
   }

   protected checkObjectPermissions(
      subrequests: CheckObjectPermissionsParams[],
   ): Observable<(EAccessControlAction | undefined)[]> {
      const data: YpApiParams[YpApiActions.CheckObjectPermissions]['request'] = {
         subrequests: subrequests.map(item => {
            const { id, login, type, permission, path } = item;
            saveYpPaths(YpApiActions.CheckObjectPermissions, type, [path ?? '']);

            return {
               object_type: type,
               object_id: id,
               subject_id: login,
               permission: permission ?? EAccessControlPermission.ACA_WRITE,
               ...(path ? { attribute_path: path } : {}),
            };
         }),
      };
      return this.apiRequest({
         action: YpApiActions.CheckObjectPermissions,
         data,
      }).pipe(map(response => response.subresponses!.map(item => item.action)));
   }

   protected updateObject<T extends UsedYpObjectTypes>({
      id,
      type,
      paths,
      prerequisites,
      location,
   }: UpdateObjectParams<T>): Observable<DeepPartial<TRspUpdateObject>> {
      saveYpPaths(YpApiActions.UpdateObject, type, Object.keys(paths));

      const input: DeepPartial<TReqUpdateObject> = {
         object_id: id,
         object_type: type,
         remove_updates: Object.entries(paths)
            .filter(([, yson]) => yson === undefined)
            .map(([path]) => ({ path })),
         set_updates: Object.entries(paths)
            .filter(([, yson]) => yson !== undefined)
            .map(([path, yson]) => ({ path, value_payload: { yson } })),
         attribute_timestamp_prerequisites: prerequisites,
      };

      return this.apiRequest({ action: YpApiActions.UpdateObject, data: input, location });
   }

   protected getAvailableIdsByLogin(
      { requests }: { requests: GetAvailableIdsByLoginParams },
      location?: YpLocation,
   ): Observable<string[][]> {
      const input: DeepPartial<TReqGetUserAccessAllowedTo> = {
         subrequests: requests.map(e => ({
            user_id: e.login ?? '',
            object_type: e.objectType,
            permission: e.permission ?? EAccessControlPermission.ACA_WRITE,
            attribute_path: e.path ?? '/spec',
            ...(e.query
               ? {
                    filter: {
                       query: e.query,
                    },
                 }
               : {}),
            continuation_token: e.continuationToken,
            limit: e.limit ?? 1000,
         })),
      };

      // TODO: на будущее предусмотреть использование токенов
      return this.apiRequest({ action: YpApiActions.GetUserAccessAllowedTo, data: input, location }).pipe(
         map(resp => resp.subresponses!.map(subresp => subresp.object_ids!)),
      );
   }

   /**
    * Загрузка постранично с токеном
    *
    * Грузит пачками, пока не достигнет limit или значения не кончатся
    */
   private loadByPage<T>({
      loader,
      continuationToken,
      prevValues = [],
      limit,
   }: LoadByPageParams<T>): Promise<LoadByPageResult<T>> {
      return loader(continuationToken).then(({ values, continuationToken: nextToken }) => {
         if (values.length === 0) {
            // значения кончились
            return Promise.resolve({
               values: prevValues,
               continuationToken: nextToken,
            });
         }

         if (prevValues.length + values.length >= limit) {
            // достигли лимита
            return Promise.resolve({
               values: prevValues.concat(values),
               continuationToken: nextToken,
            });
         }

         // следующая пачка
         return this.loadByPage({
            loader,
            continuationToken: nextToken,
            prevValues: prevValues.concat(values),
            limit,
         });
      });
   }

   // TODO: добавить null
   private parseYpResponse<R extends UsedYpObjects>(
      rawResult: DeepPartial<TAttributeList>,
      paths: string[] | undefined,
   ): DeepPartial<R> {
      const result = rawResult as YpAttributeList;

      // Нужно восстановить структуру объекта.
      const values = result.value_payloads.map(e => e.yson);
      const updatePaths: (string | symbol)[][] = (paths ?? ['']).map(e => [e]);

      if (result.timestamps) {
         // пишем время в отдельное символьное поле
         values.push(...result.timestamps.map(e => Number(parseYpTimestamp(e))));
         updatePaths.push(...updatePaths.map(e => [...e, Symbol.for('timestamp')]));
      }
      return this.restoreObjectByYpPaths<R>(values, updatePaths);
   }

   protected parseError(body: any, resp: any): ApiErrorItemWithData<NetworkYpErrorData> {
      const ytErrorStr: string | undefined = resp.headers.get('X-YT-Error');
      const ytMessage: string | undefined = resp.headers.get('X-YT-Response-Message');
      const ytTraceId: string | undefined = resp.headers.get('X-YT-Trace-Id');
      const result: ApiErrorItemWithData<NetworkYpErrorData> = { body, resp };
      const data: NetworkYpErrorData = {
         ytError: ytErrorStr ? JSON.parse(ytErrorStr) : undefined,
         ytMessage,
         ytTraceId,
      };
      if (Object.values(data).some(Boolean)) {
         result.data = data;
      }

      return result;
   }

   private apiRequest<Action extends YpApiActions>({
      action,
      data,
      location = YpLocation.XDC,
      headers: customHeaders,
   }: ApiRequest<ApiServiceName.YP, Action, 'request'> & YpRequestAdditionalParams): Observable<
      YpApiParams[Action]['response']
   > {
      const { method, path, credentials = 'include', headers: serviceHeaders } = getApiRequestParams({
         service: ApiServiceName.YP,
         action,
      });
      const endpoint = this.endpoints[location] ?? this.endpoints[YpLocation.XDC]!;

      const headers: Record<string, string> = { ...serviceHeaders, ...customHeaders };

      const url = endpoint + path;

      type CurrentMetaData = RequestMetaData<ApiServiceName.YP, Action, YpApiParams[Action]['request']>;

      const requestMetaData: CurrentMetaData = {
         service: ApiServiceName.YP,
         action,
         requestData: data,
         url,
         headers,
      };

      const request = this.request<any, YpApiParams[Action]['request'], YpApiParams[Action]['response']>({
         method,
         path: url,
         params: this.devMode
            ? {
                 t: (data as any).object_type,
              }
            : undefined,
         body: data,
         headers,
         credentials,
         onlyJson: false,
      }).pipe(
         map(body => {
            // TODO зачем этот map() вообще, если можно не передавать onlyJson:false ?
            if (typeof body === 'string') {
               return JSON.parse(body);
            }
            return body;
         }),
         catchError((error: ApiErrorItemWithData<NetworkYpErrorData> | Error) => {
            throw prepareApiError({ error, requestMetaData, errorName: 'Yp Api Error' });
         }),
      ) as Observable<YpApiParams[Action]['response']>;

      (request as any)[Symbol.for('requestMeta')] = requestMetaData;

      return request;
   }

   private restoreObjectByYpPaths<T extends Record<string, any>>(
      values: any[],
      paths: (string | symbol)[][],
   ): DeepPartial<T> {
      return values.reduce((acc, value, i) => updateYpObject(acc, paths[i], value), {} as DeepPartial<T>);
   }
}
