type GetPaths<T> = (() => PathData) & {
   (name: any): GetPaths<T extends (infer X)[] ? X : T[keyof T]>;
} & {
      [name in keyof T]: GetPaths<T[name]>;
   };

type PathData = Array<{ type: 'const' | 'let'; value: string }>;

export function getPaths<T>(pathData: PathData = []): GetPaths<T> {
   return new Proxy(() => pathData as any, {
      get(_, name) {
         return getPaths<T[keyof T]>([...pathData, { type: 'const', value: name as string }]);
      },
      apply(_, __, args: [] | [any]) {
         if (args.length === 0) {
            return pathData;
         }
         return getPaths<T[keyof T]>([...pathData, { type: 'let', value: String(args[0]) as string }]);
      },
   }) as GetPaths<T>;
}

export interface PathOptions {
   separators?: {
      constLeft?: string;
      constRight?: string;
      letLeft?: string;
      letRight?: string;
   };
   hideSeparators?: {
      firstConstLeft?: boolean;
      firstLetLeft?: boolean;
      lastConstRight?: boolean;
      lastLetRight?: boolean;
   };
}

export const pathBuilder = (options: PathOptions) => (getPathData: () => PathData) => {
   const { constLeft = '.', constRight = '', letLeft = ':', letRight = '' } = options.separators ?? {};
   const { firstConstLeft = false, firstLetLeft = true, lastConstRight = false, lastLetRight = true } =
      options.hideSeparators ?? {};
   return getPathData()
      .map(({ type, value }, i, arr) => {
         const left = { const: constLeft, let: letLeft }[type];
         const right = { const: constRight, let: letRight }[type];
         const hideFirst = { const: firstConstLeft, let: firstLetLeft }[type];
         const hideLast = { const: lastConstRight, let: lastLetRight }[type];
         return `${i === 0 && hideFirst ? '' : left}${value}${i === arr.length - 1 && hideLast ? '' : right}`;
      })
      .join('');
};
