import { getSetDifference } from '@yandex-infracloud-ui/libs';

export function createKey(object: Record<string, string>): string {
   const params = [];
   // сортируем, чтобы строки совпадали
   for (const key of Object.keys(object).sort()) {
      params.push({ [key]: object[key] });
   }
   return JSON.stringify(params);
}

export function restoreObjectFromKey(key: string): Record<string, string> {
   const object: Record<string, string> = {};
   let list: Record<string, string>[];
   try {
      list = JSON.parse(key);
   } catch (error) {
      console.error(`Format for key is Record<string, string>[]`);
      throw error;
   }
   for (const pair of list) {
      const [name, value] = Object.entries(pair)[0];
      object[name] = value;
   }
   return object;
}

/**
 * Действие над хранилищем, которое может формироваться и передаваться независимо от него
 *
 * T - тип значения в хранилище
 */
export interface StoreAction<T> {
   /**
    * удалить указанные ключи, если они есть, доступно удаление по условию
    */
   delete: (string | { condition?: (oldValue: T | undefined) => boolean; key: string })[];
   /**
    * обновить значения по ключам либо записать новые, доступно условие записи и геттер
    */
   write: Record<
      string,
      {
         condition?: (oldValue: T | undefined, newValue: T) => boolean;
         value?: T;
         getValue?: (oldValue: T | undefined) => T;
      }
   >;
}

export function getEmptyStoreAction<T>(): StoreAction<T> {
   return {
      delete: [],
      write: {},
   };
}

export function isEmptyStoreAction<A extends StoreAction<any>>(action: A): boolean {
   const isEmptyDelete = action.delete.length === 0;
   const isEmptyWrite = Object.keys(action.write).length === 0;
   return isEmptyDelete && isEmptyWrite;
}

export function updateStore<T>(store: Record<string, T>, action: StoreAction<T>): Record<string, T> {
   const editedStore = { ...store };
   for (const item of action.delete) {
      const key = typeof item === 'string' ? item : item.key;
      const needDelete = typeof item === 'string' || (item.condition ? item.condition(editedStore[key]) : true);
      if (needDelete && editedStore.hasOwnProperty(key)) {
         delete editedStore[key];
      }
   }
   for (const key of Object.keys(action.write)) {
      const { value, condition, getValue } = action.write[key];
      const newValue = value ?? (getValue ? getValue(editedStore[key]) : undefined);
      const needWrite = newValue !== undefined && (condition ? condition(editedStore[key], newValue) : true);
      if (newValue !== undefined && needWrite && editedStore[key] !== newValue) {
         editedStore[key] = newValue;
      }
   }
   return editedStore;
}

export function mergeStores<T1, T2, U>(
   store1: Record<string, T1>,
   store2: Record<string, T2>,
   map: (a: T1 | undefined, b: T2 | undefined) => U,
): Record<string, U> {
   const store: Record<string, U> = {};
   const unionKeys = new Set([...Object.keys(store1), ...Object.keys(store2)]);
   for (const key of unionKeys.values()) {
      store[key] = map(store1[key], store2[key]);
   }
   return store;
}

export type Tree<Order extends [string, ...string[]], T> = Partial<{
   [key: string]: Order extends [string]
      ? T
      : Order extends [string, ...infer X]
      ? X extends [string, ...string[]]
         ? Tree<X, T>
         : never
      : never;
}>;

export function buildTree<T, Order extends [string, ...string[]]>(
   store: Record<string, T>,
   order: Order,
): Tree<Order, T> {
   const currentName = order[0];
   const fieldsSet = new Set(order);
   const tree: Tree<Order, T> = {};
   const currentValues: Record<string, Record<string, T>> = {};

   for (const key of Object.keys(store)) {
      const targetValue = store[key];
      const fields = restoreObjectFromKey(key);
      const names = Object.keys(fields);

      const nameSet = new Set(names);
      const { removed, added } = getSetDifference(fieldsSet, nameSet);

      if (added.size > 0) {
         throw new Error(`Fields ${[...added.values()]} dont exist in order ${order}`);
      }
      if (removed.size > 0) {
         console.dir(store);
         throw new Error(`Order fields [${[...removed.values()].join(',')}] dont exist in fields ${names}`);
      }

      const currentValue = fields[currentName];
      if (!currentValues.hasOwnProperty(currentValue)) {
         currentValues[currentValue] = {};
      }
      const restFields = { ...fields };
      delete restFields[currentName];

      currentValues[currentValue][createKey(restFields)] = targetValue;
   }

   for (const value of Object.keys(currentValues)) {
      if (order.length === 1) {
         const [targetValue] = Array.from(Object.values(currentValues[value]));
         tree[value] = targetValue as any;
      } else {
         tree[value] = buildTree(currentValues[value], order.slice(1) as any) as any;
      }
   }

   return tree;
}

const keysSymbol = Symbol.for('keys');
type keysType = typeof keysSymbol;
export type StoreKeysMeta<T extends string> = { [keys in keysType]?: T };
