import { formatNumber } from '@yandex-infracloud-ui/libs';

export enum MeasureViewMode {
   All = 'all',
   OnlyValue = 'onlyValue',
   OnlyUnit = 'onlyUnit',
}

export enum MeasureUnitMode {
   /** Приведение к максимально близкому разряду */
   Auto = 'auto',
   /** Приведение к указанному разряду */
   Custom = 'custom',
   /** Полная цепочка разрядов вплоть до указанного разряда */
   Chain = 'chain',
}

export interface MeasureFormatOptions<UnitName extends string> {
   /** округление с точностью до `precision` либо отсутствие округления, по умолчанию `2` */
   precision?: number | null;
   /** Настройка вывода, по умолчанию `MeasureViewMode.All` */
   viewMode?: MeasureViewMode;
   /** Настройка приведения к разрядам, по умолчанию `MeasureUnitMode.Auto` */
   unitMode?: MeasureUnitMode;
   /** Только для `MeasureUnitMode.Custom` и `MeasureUnitMode.Chain` */
   unit?: UnitName;
   /** функция округления (если `precision !== null`), по умолчанию `Math.round` */
   roundFunction?: (n: number) => number;
}

export function getPositionalUnitMeasures<Unit extends string>(
   list: readonly [Unit, ...Unit[]],
   multiplier: number,
): Record<Unit, number> {
   return list.reduce((obj, unit, i) => {
      obj[unit] = multiplier ** i;
      return obj;
   }, {} as Record<Unit, number>);
}

function getValueView(
   value: number,
   unit: string,
   viewMode: MeasureViewMode,
   precision: number | null,
   roundFunction: (n: number) => number,
): string {
   let result = value;
   const targetPrecision = precision && !isNaN(precision) ? precision : 0;
   const existPrecision = targetPrecision > 0;

   if (existPrecision) {
      const exp = Math.pow(10, targetPrecision);
      result = roundFunction(result * exp) / exp;
   }

   const additionalFormat = existPrecision ? `.[${Array(targetPrecision).fill(0).join('')}]` : '';

   const resultView = formatNumber(result, `0,0${additionalFormat}`);

   return {
      [MeasureViewMode.OnlyUnit]: unit,
      [MeasureViewMode.OnlyValue]: resultView,
      [MeasureViewMode.All]: `${resultView}\u00a0${unit}`,
   }[viewMode];
}

export function formatMeasureValue<UnitName extends string>(
   value: number,
   measures: Record<UnitName, number>,
   options?: MeasureFormatOptions<UnitName>,
) {
   const {
      precision: customPrecision,
      viewMode = MeasureViewMode.All,
      unitMode = MeasureUnitMode.Auto,
      unit: customUnit,
      roundFunction = Math.round,
   } = options ?? {};
   const measureList = (Object.entries(measures) as [UnitName, number][]).sort((a, b) => b[1] - a[1]);

   const unit = customUnit ?? measureList[0][0];
   const precision = customPrecision === undefined ? 2 : customPrecision;

   const getView = (v: number, u: UnitName) => getValueView(v, u, viewMode, precision, roundFunction);

   if (unitMode === MeasureUnitMode.Custom) {
      const multiplier = measures[unit];
      const measureValue = value / multiplier;

      return getView(measureValue, unit);
   }

   let result = value;
   const chain: [number, string][] = [];

   for (const [currentUnit, multiplier] of measureList) {
      if (customUnit !== undefined && currentUnit === customUnit) {
         const roundedValue = result / multiplier;

         chain.push([roundedValue, getView(roundedValue, customUnit)]);
         break;
      }

      const part = result % multiplier;
      const roundedValue = (result - part) / multiplier;

      chain.push([roundedValue, getView(roundedValue, currentUnit)]);

      result = part;
   }

   const valuesChain = chain.filter(e => e[0] > 0).map(e => e[1]);
   const zeroValue = valuesChain.length === 0 ? chain.find(e => e[0] === 0)?.[1] : null;

   if (unitMode === MeasureUnitMode.Chain) {
      return zeroValue ?? valuesChain.join(' ');
   }
   return zeroValue ?? valuesChain[0];
}
