import { useCallback, useEffect, useRef, useState } from 'react';

/**
 * Уничтожает компонент на некоторое время, форсируя его полную переинициализацию
 *
 * Разумеется, это костыль, и применять следует только в крайних случаях, например,
 * для библиотечных компонентов, которые редактировать мы не можем.
 *
 * В большинстве случаев лучше использовать useForceRender (что тоже крайняя мера, но не настолько костыльная)
 */
export function useForceRecreate(defaultDelay?: number) {
   const [show, setShow] = useState(true);
   const timeoutId = useRef(0);
   const rafRef = useRef(0);

   const recreate = useCallback(
      (delay?: number) => {
         setShow(false);
         const d = delay ?? defaultDelay;

         if (d) {
            window.setTimeout(() => setShow(true), d);
         } else {
            rafRef.current = window.requestAnimationFrame(() => setShow(true));
         }
      },
      [defaultDelay],
   );

   useEffect(() => () => {
      window.clearTimeout(timeoutId.current);
      window.cancelAnimationFrame(rafRef.current);
   });

   return { show, recreate };
}
