import { useEffect, useRef, useState } from 'react';
import { noop } from '../helpers';

/**
 * На один тик прячет компонент, заставляя его переинициализироваться
 *
 * Это безусловно костыль. Он нужен, чтобы обойти сохранение локального стейта при изменении роута:
 * react-router не перерендеривает компонент-страницу, если сам роут не поменялся, а поменялся только location.
 * Он просто передаёт в компонент новые пропсы (location, match).
 *
 * Иногда это поведение мешает, и этот хук позволяет его обойти.
 *
 * @example
 *    // Где-то перед обычным return с JSX
 *    const visible = useLocationChangeRender(location.pathname);
 *    if (!visible) {
 *       return null;
 *    }
 *
 *    return <div>bla-bla</div>
 *
 * @param pathname = props.location.pathname, то, изменение чего будет приводить к ререндеру
 */
export function useLocationChangeRender(pathname: string) {
   const firstRef = useRef(true);
   const [visible, setVisible] = useState(true);

   useEffect(() => {
      // Пропуск при первом рендере
      if (firstRef.current) {
         firstRef.current = false;
         return noop;
      }

      setVisible(false);

      const requestId = window.requestAnimationFrame(() => setVisible(true));

      return () => window.cancelAnimationFrame(requestId);
   }, [pathname]);

   return visible;
}
