import { Dispatch, SetStateAction, useMemo, useState } from 'react';

type Tail<TState extends Object, T extends any[]> = T extends [TState, ...infer X] ? X : [];

type Updaters<TState extends Object> = Record<string, (prevState: TState, ...params: any[]) => TState>;

type WrappedUpdaters<TState extends Object, TUpdaters extends Updaters<TState>> = {
   [m in keyof TUpdaters]: (...args: Tail<TState, Parameters<TUpdaters[m]>>) => TState;
};

// Creates wrapper functions for original updaters and pass current state and arguments to original updaters
function wrapUpdaters<TState extends Object, TUpdaters extends Updaters<TState>>(
   rawUpdaters: TUpdaters,
   updateState: Dispatch<SetStateAction<TState>>,
) {
   return Object.entries(rawUpdaters).reduce((acc: any, [method, updater]) => {
      acc[method] = (...params: any[]) => {
         updateState(prevState => updater(prevState, ...params));
      };

      return acc;
   }, {} as WrappedUpdaters<TState, TUpdaters>);
}

/**
 * @see https://github.com/reactjs/rfcs/issues/185#issuecomment-764866437
 * @example
 * interface CounterState {
 *   count: number;
 * }
 *
 * const updaters = {
 *   subtract: (prevState: CounterState, value: number) => ({
 *     ...prevState,
 *     count: prevState.count - value
 *   }),
 *   add: (prevState: CounterState, value: number) => ({
 *     ...prevState,
 *     count: prevState.count + value
 *   })
 * };
 *
 * const MyComponent = () => {
 *  const [{ count }, { subtract, add }] = useStateWithUpdaters(
 *    { count: 0 },
 *    updaters
 *  );
 *  return (
 *    <div>
 *      Count: {count}
 *      <button onClick={() => subtract(1)}>-</button>
 *      <button onClick={() => add(1)}>+</button>
 *    </div>
 *  );
 * };
 */
export function useStateWithUpdaters<TState extends Object, TUpdaters extends Updaters<TState>>(
   defaultState: TState,
   rawUpdaters: TUpdaters,
): [TState, WrappedUpdaters<TState, TUpdaters>] {
   const [state, updateState] = useState(defaultState);
   const updaters = useMemo(() => wrapUpdaters(rawUpdaters, updateState), [rawUpdaters]);

   return [state, updaters];
}
