import { parseTextToObjects, TextParserType } from './parseTextToObjects';

describe('utils|parsers|parseTextToObjects', () => {
   describe('parseTextToObjects', () => {
      it('empty string, should return empty array', () => {
         expect(parseTextToObjects('').length).toBe(1);
      });

      it('should return array with one element (default text) on empty string without flags', () => {
         const result = parseTextToObjects(' ');
         expect(result.length).toBe(1);
         expect(result[0].type).toBe(TextParserType.DEFAULT_TEXT);
      });

      it('should return one element (default text) on string with flags', () => {
         const result = parseTextToObjects('some text https://ya.ru DEPLOY-333 some text', {
            links: false,
            startrekTicketUrl: false,
            startrekTicket: false,
         });
         expect(result.length).toBe(1);
         expect(result[0].type).toBe(TextParserType.DEFAULT_TEXT);
      });

      it('should return one element ', () => {
         const result = parseTextToObjects('hello, my text here!', {
            links: true,
            startrekTicketUrl: true,
            startrekTicket: true,
         });
         expect(result.length).toBe(1);
         expect(result[0].type).toBe(TextParserType.DEFAULT_TEXT);
      });

      it('should return five elements, 3 with default, 2 with links', () => {
         const result = parseTextToObjects(
            'some text http://ya.ru/test/data?intag=333 some text https:/not-correct-link, https://habr.com/fef ',
            {
               links: true,
               startrekTicket: true,
               startrekTicketUrl: true,
            },
         );

         expect(result.length).toBe(5);

         expect(result[0].type).toBe(TextParserType.DEFAULT_TEXT);
         expect(result[0].text).toBe('some text ');

         expect(result[1].type).toBe(TextParserType.LINK);
         expect(result[1].text).toBe('http://ya.ru/test/data?intag=333');

         expect(result[2].type).toBe(TextParserType.DEFAULT_TEXT);
         expect(result[2].text).toBe(' some text https:/not-correct-link, ');

         expect(result[3].type).toBe(TextParserType.LINK);
         expect(result[3].text).toBe('https://habr.com/fef');

         expect(result[4].type).toBe(TextParserType.DEFAULT_TEXT);
         expect(result[4].text).toBe(' ');
      });

      it('should parse links and startrek tags only', () => {
         const result = parseTextToObjects(
            'TICKET-1333 https://st.yandex-team.ru/tickets/TICKET-15555 some-text in the end',
            {
               links: true,
               startrekTicket: true,
               startrekTicketUrl: false,
            },
         );

         expect(result.length).toBe(4);

         expect(result[0].type).toBe(TextParserType.STARTREK_TICKET);
         expect(result[0].text).toBe('TICKET-1333');

         expect(result[1].type).toBe(TextParserType.DEFAULT_TEXT);
         expect(result[1].text).toBe(' ');

         expect(result[2].type).toBe(TextParserType.LINK);
         expect(result[2].text).toBe('https://st.yandex-team.ru/tickets/TICKET-15555');

         expect(result[3].type).toBe(TextParserType.DEFAULT_TEXT);
         expect(result[3].text).toBe(' some-text in the end');
      });

      it('incorrect ticket name', () => {
         const result = parseTextToObjects('some text hello asSPI-16088-logbroker-yd-throttling-rollback', {
            links: true,
            startrekTicket: true,
            startrekTicketUrl: true,
         });

         expect(result.length).toBe(1);

         expect(result[0].type).toBe(TextParserType.DEFAULT_TEXT);
         expect(result[0].text).toBe('some text hello asSPI-16088-logbroker-yd-throttling-rollback');
      });

      it('parse all tickets correct', () => {
         const result = parseTextToObjects(
            'TICKET-3333 TICKET-123, asdasd,asd asd,asd a TICKET-5555,sadasdsaTICKET-333asdas, TICKET-111',
            {
               links: true,
               startrekTicket: true,
               startrekTicketUrl: true,
            },
         );

         expect(result.length).toBe(7);

         expect(result[0].type).toBe(TextParserType.STARTREK_TICKET);
         expect(result[0].text).toBe('TICKET-3333');

         expect(result[1].type).toBe(TextParserType.DEFAULT_TEXT);
         expect(result[1].text).toBe(' ');

         expect(result[2].type).toBe(TextParserType.STARTREK_TICKET);
         expect(result[2].text).toBe('TICKET-123');

         expect(result[3].type).toBe(TextParserType.DEFAULT_TEXT);
         expect(result[3].text).toBe(', asdasd,asd asd,asd a ');

         expect(result[4].type).toBe(TextParserType.STARTREK_TICKET);
         expect(result[4].text).toBe('TICKET-5555');

         expect(result[5].type).toBe(TextParserType.DEFAULT_TEXT);
         expect(result[5].text).toBe(',sadasdsaTICKET-333asdas, ');

         expect(result[6].type).toBe(TextParserType.STARTREK_TICKET);
         expect(result[6].text).toBe('TICKET-111');
      });

      it('parse single ticket', () => {
         const result = parseTextToObjects('DEPLOY-3070', {
            links: true,
            startrekTicket: true,
            startrekTicketUrl: true,
         });

         expect(result.length).toBe(1);

         expect(result[0].type).toBe(TextParserType.STARTREK_TICKET);
         expect(result[0].text).toBe('DEPLOY-3070');
      });

      it('parse single link', () => {
         const result = parseTextToObjects('https://yandex.ru/hits', {
            links: true,
            startrekTicket: true,
            startrekTicketUrl: true,
         });

         expect(result.length).toBe(1);

         expect(result[0].type).toBe(TextParserType.LINK);
         expect(result[0].text).toBe('https://yandex.ru/hits');
      });

      it('parse single ticket link', () => {
         const result = parseTextToObjects('https://st.yandex-team.ru/hits/DEPLOY-555', {
            links: true,
            startrekTicket: true,
            startrekTicketUrl: true,
         });

         expect(result.length).toBe(1);

         expect(result[0].type).toBe(TextParserType.STARTREK_TICKET);
         expect(result[0].text).toBe('DEPLOY-555');
      });
   });
});
