export enum TextParserType {
   STARTREK_TICKET = 'STARTREK_TICKET', // format: DEPLOY-333 or http(s)://st.yandex-team.ru/some/url/TICKET-333
   LINK = 'LINK', // format: any url
   DEFAULT_TEXT = 'DEFAULT_TEXT', // other text
}

export type TextParserResult = {
   text: string;
   type: TextParserType;
};

type TextParserMode = {
   links?: boolean; // парсим ли мы ссылки
   startrekTicket?: boolean; // парсим ли мы голые тикеты (DEPLOY-333)
   startrekTicketUrl?: boolean; // парсим ли мы ссылки на тикеты! в голый тикет
};

type TextNode = {
   text: string;
   wasModify: boolean; // было ли поле text уже обработано
};

function parseEachStringByPattern(
   inputString: TextNode[],
   pattern: RegExp,
   patternInSubstring: RegExp | null = null,
): TextNode[] {
   const parseResult: TextNode[] = [];

   for (const { text, wasModify } of inputString) {
      const matchResult = text.matchAll(pattern);

      /*
         разбиваем text на куски и возвращаем
       */
      let start = 0;

      if (!wasModify) {
         for (const elemInMatch of matchResult) {
            const { index } = elemInMatch;
            const substring = elemInMatch[0];

            if (index !== undefined && substring) {
               let insertingSubstring = substring;
               if (patternInSubstring !== null) {
                  const result = insertingSubstring.match(patternInSubstring);

                  if (Array.isArray(result) && result.length > 0) {
                     [insertingSubstring] = result;
                  }
               }

               parseResult.push(
                  {
                     text: text.substring(start, index),
                     wasModify: false,
                  },
                  {
                     text: insertingSubstring,
                     wasModify: true,
                  },
               );

               start = index + substring.length;
            }
         }
      }

      if (start < text.length) {
         parseResult.push({
            text: text.substring(start),
            wasModify: false,
         });
      }
   }

   return parseResult.filter(({ text }) => text.length > 0);
}

/**
 * Парсим строку в массив объектов, согласно правилам включенных модов
 * Пример:
 * строка - "Hello, world! https://yandex.ru, DEPLOY-123, https://st.yandex-team.ru/xxxx/DEPLOY-555 some message "
 * со всеми вкл модами получим объект
 * [
 *    {
 *       text: "Hello, world! ",
 *       type: "DEFAULT_TEXT",
 *    },
 *    {
 *       text: "https://yandex.ru",
 *       type: "URL",
 *    },
 *    {
 *       text: ", ",
 *       type: "DEFAULT_TEXT",
 *    },
 *    {
 *       text: "DEPLOY-123",
 *       type: "STARTREK_TICKET",
 *    },
 *    {
 *       text: ", ",
 *       type: "DEFAULT_TEXT",
 *    },
 *    {
 *       text: "DEPLOY-555",
 *       type: "STARTREK_TICKET",
 *    },
 *    {
 *       text: " some message ",
 *       type: "DEFAULT_TEXT",
 *    },
 * ]
 * Параметры:
 * @param str {string} исходная строка
 * mode object:
 * @param links {boolean} парсим URL вида http(s)://
 * @param startrekTicket  {boolean}  парсим тикеты из Стартрека вида DEPLOY-333
 * @param startrekTicketUrl  {boolean} парсим URL тикеов из Стретка в отдельный тикет
 */
export function parseTextToObjects(
   str: string,
   { links = false, startrekTicket = false, startrekTicketUrl = false }: TextParserMode = {},
): TextParserResult[] {
   const result: TextParserResult[] = [];

   const STARTREK_TICKET_PATTERN = /(\b[A-Z]{2,15}-[0-9]{1,12}\b)/g;
   const STARTREK_URL_PATTERN = /(\b(https?):\/\/st\.yandex-team\.ru\/[a-z/A-Z0-9-#%$;=?&'"]*[-A-Z0-9+&@#/%=~_|]\/[A-Za-z]{2,10}-[0-9]{1,12})/gi;
   const LINKS_PATTERN = /(\b(https?|ftp|file):\/\/[-A-Z0-9+&@#/%?=~_|!:,.;]*[-A-Z0-9+&@#/%=~_|])/gi;

   // ничего не хотим искать, то возвращаем исходную строку
   const emptySearch = !links && !startrekTicket && !startrekTicketUrl;
   if (emptySearch || !str || str.length < 1) {
      return [
         {
            text: ``,
            type: TextParserType.DEFAULT_TEXT,
         },
      ];
   }

   let initialText: TextNode[] = [
      {
         text: `${str}`,
         wasModify: false,
      },
   ];

   // Parsing flow
   const patterns = [];

   if (startrekTicketUrl) {
      patterns.push({
         pattern: STARTREK_URL_PATTERN,
         substringPattern: STARTREK_TICKET_PATTERN,
      });
   }

   if (links) {
      patterns.push({
         pattern: LINKS_PATTERN,
      });
   }

   if (startrekTicket) {
      patterns.push({
         pattern: STARTREK_TICKET_PATTERN,
      });
   }

   for (const { pattern, substringPattern = null } of patterns) {
      initialText = parseEachStringByPattern(initialText, pattern, substringPattern);
   }

   const parseStartrek = startrekTicket || startrekTicketUrl;

   // превращаем в объект
   for (const { text: elem } of initialText) {
      const textParserElem: TextParserResult = {
         text: elem,
         type: TextParserType.DEFAULT_TEXT,
      };
      if (links && LINKS_PATTERN.test(elem)) {
         textParserElem.type = TextParserType.LINK;
      } else if (parseStartrek && STARTREK_TICKET_PATTERN.test(elem)) {
         textParserElem.type = TextParserType.STARTREK_TICKET;
      }

      result.push(textParserElem);
   }

   return result;
}
