import { recalculateCounts } from './recalculateCounts';

const getMap = <K extends string, T>(record: Record<K, T>) => new Map(Object.entries(record)) as Map<K, T>;
const getRecord = <T>(map: Map<string, T>) => Object.fromEntries(map.entries());

describe('recalculate counts', () => {
   it('actual counts', () => {
      // все актуальные значения скачаны полностью
      const visible = getMap({ red: 2, green: 2, blue: 2 });
      const actual = getMap({ red: 1, green: 1, blue: 0 });
      const loadAll = getMap({ red: true, green: true, blue: true });

      const { result } = recalculateCounts({ visible, actual, loadAll });

      expect(getRecord(result)).toEqual({ red: 1, green: 1, blue: 0 });
   });

   it('large visible', () => {
      // скачана только маленькая часть видимых множеств
      const visible = getMap({ red: 2000, green: 2000, blue: 2000 });
      const actual = getMap({ red: 10, green: 10, blue: 10 });
      const loadAll = getMap({ red: false, green: false, blue: false });

      const { result } = recalculateCounts({ visible, actual, loadAll });

      expect(getRecord(result)).toEqual({ red: 2000, green: 2000, blue: 2000 });
   });

   it('large actual', () => {
      // скачано больше объектов, чем видимо
      const visible = getMap({ red: 10, green: 10, blue: 10 });
      const actual = getMap({ red: 50, green: 50, blue: 50 });
      const loadAll = getMap({ red: false, green: false, blue: false });

      const { result } = recalculateCounts({ visible, actual, loadAll });

      expect(getRecord(result)).toEqual({ red: 50, green: 50, blue: 50 });
   });

   it('large actual part', () => {
      // скачано больше объектов одного типа, чем видимо всех остальных, и объекты этого типа забивают всё
      const visible = getMap({ red: 10, green: 10, blue: 10 });
      const actual = getMap({ red: 50, green: 0, blue: 0 });
      const loadAll = getMap({ red: false, green: false, blue: false });

      const { result } = recalculateCounts({ visible, actual, loadAll });

      expect(getRecord(result)).toEqual({ red: 50, green: 0, blue: 0 });
   });

   it('large actual part, large visible part', () => {
      // скачано много объектов одного типа, но видимо много объектов другого типа, и они забирают доступные места
      const visible = getMap({ red: 10, green: 20, blue: 0 });
      const actual = getMap({ red: 20, green: 0, blue: 0 });
      const loadAll = getMap({ red: false, green: false, blue: false });

      const { result } = recalculateCounts({ visible, actual, loadAll });

      expect(getRecord(result)).toEqual({ red: 20, green: 10, blue: 0 });
   });

   it('large actual part, large visible part, load all part 1', () => {
      // скачано много объектов одного типа, но видимо много объектов другого типа, и они забирают доступные места
      // но так как скачанны уже все зеленые объекты, остаток распределяется по красным и синим
      const visible = getMap({ red: 10, green: 20, blue: 0 });
      const actual = getMap({ red: 20, green: 0, blue: 0 });
      const loadAll = getMap({ red: false, green: true, blue: false });

      const { result } = recalculateCounts({ visible, actual, loadAll });

      expect(getRecord(result)).toEqual({ red: 25, green: 0, blue: 5 });
   });

   it('large actual part, large visible part, load all part 2', () => {
      // скачано много объектов одного типа, но видимо много объектов другого типа, и они забирают доступные места
      // но так как скачанны уже все зеленые и синие объекты, остаток распределяется по красным
      const visible = getMap({ red: 0, green: 10, blue: 20 });
      const actual = getMap({ red: 0, green: 20, blue: 0 });
      const loadAll = getMap({ red: false, green: true, blue: true });

      const { result } = recalculateCounts({ visible, actual, loadAll });

      expect(getRecord(result)).toEqual({ red: 10, green: 20, blue: 0 });
   });
});
