export type CountMap<T extends string = string> = Map<T, number>;
type CountPair<T extends string = string> = [T, number];
type CountList<T extends string = string> = CountPair<T>[];

export type RecalculateCountsParams<T extends string> = {
   visible: CountMap<T>;
   actual: CountMap<T>;
   loadAll: Map<T, boolean>;
};

export type RecalculateCountsResult<T extends string> = {
   result: CountMap<T>;
};

/**
 * Пересчитывает количество объектов в группах для соответствия ограничениям
 * @param params.visible Видимые группы, их значения могут быть не достоверны
 * @param params.actual Полученные части групп, их значения полностью достоверны (ограничение снизу)
 * @param params.loadAll Флаги загрузки, для группы выставлен флаг, если она скачена полностью (ограничение сверху)
 * @returns Смерженные количества групп с учётом ограничений
 */
export function recalculateCounts<T extends string>(params: RecalculateCountsParams<T>): RecalculateCountsResult<T> {
   const { visible, actual, loadAll } = params;
   const visibleSum = getSum(visible);
   const actualSum = getSum(actual);

   const result: CountMap<T> = new Map();
   const freeze: Set<T> = new Set();
   const visibleRest: CountMap<T> = new Map();
   const allNames = new Set([...visible.keys(), ...actual.keys()]);
   for (const group of allNames.values()) {
      const visibleCount = visible.get(group) ?? 0;
      const actualCount = actual.get(group) ?? 0;
      const isAll = loadAll.get(group) ?? false;

      // записываем то, что точно известно
      result.set(group, actualCount);
      if (isAll) {
         // если группа скачана целиком, то менять её больше нельзя
         freeze.add(group);
         // eslint-disable-next-line no-continue
         continue;
      }

      // иначе запоминаем расхождение с видимым значением (сколько не хватает)
      visibleRest.set(group, Math.max(0, visibleCount - actualCount));
   }

   // набрали больше, уходим
   if (actualSum > visibleSum) {
      return { result };
   }

   // доступные группы для изменения
   const availableGroups: CountList<T> = Array.from(visibleRest).filter(e => !freeze.has(e[0]));

   if (availableGroups.length === 0) {
      return { result };
   }

   // общий остаток
   const restSum = visibleSum - actualSum;

   // раскидываем остаток по доступным группам
   addRest({ result, availableGroups, restSum });

   return { result };
}

/**
 * Распределение остатка по группам
 */
function addRest({
   result,
   restSum,
   availableGroups,
}: {
   result: CountMap;
   restSum: number;
   availableGroups: CountList;
}): void {
   // список доступных мест по группам от большего к меньшему
   const groupList = getList(availableGroups);

   // либо остаток распределён, либо доступные местам заполнены
   while (restSum > 0 && groupList.length > 0) {
      const size = groupList.length;
      const [, minCount] = groupList[size - 1];
      if (minCount === 0) {
         // выкидываем заполненные группы
         groupList.pop();
         // eslint-disable-next-line no-continue
         continue;
      }

      // получаем распределение по группам
      const { additional, sum, listValues } = getFullGroups({ groupList, restSum, minCount });

      // обновляем остаток
      restSum = sum;

      // обновляем набранное количество
      updateCounts({ target: result, additional });

      // обновляем доступное место
      updateGroupList({ target: groupList, listValues });
   }

   // все доступное место занято, а остаток не кончился
   if (restSum > 0) {
      const allGroupList = getList(availableGroups);

      // распределяем остаток без ограничений сверху - равномерно размазываем его по группам
      const { additional } = getFullGroups({ groupList: allGroupList, restSum, minCount: +Infinity });

      // обновляем количество
      updateCounts({ target: result, additional });
   }
}

/**
 * Распределение части остатка по группам с ограничением сверху
 */
function getFullGroups<T extends string>({
   groupList,
   restSum,
   minCount,
}: {
   groupList: CountList<T>;
   restSum: number;
   minCount: number;
}): { additional: CountMap<T>; sum: number; listValues: CountMap } {
   const result: CountMap<T> = new Map();

   // полностью копируем для неизменяемости
   const list = groupList.map(([group, count]) => [group, count] as CountPair<T>);
   const listValues: CountMap = new Map(list);

   const size = list.length;

   // среднее целое по группам
   const averageCount = (restSum - (restSum % size)) / size;

   // среднее с учётом ограничения
   const availableCount = Math.min(averageCount, minCount);

   if (availableCount > 0) {
      // хватило на все группы
      for (const item of list) {
         listValues.set(item[0], item[1] - availableCount);
         result.set(item[0], (result.get(item[0]) ?? 0) + availableCount);
         restSum -= availableCount;
      }
   } else {
      // остатка хватит только на несколько первых групп, отсыпаем им по единице
      for (const item of list) {
         listValues.set(item[0], item[1] - 1);
         result.set(item[0], (result.get(item[0]) ?? 0) + 1);
         restSum -= 1;
         if (restSum === 0) {
            break;
         }
      }
   }

   return { additional: result, sum: restSum, listValues };
}

/**
 * обновление карты значений, добавление новых
 */
function updateCounts({ target, additional }: { target: CountMap; additional: CountMap }): void {
   for (const [group, count] of additional) {
      target.set(group, (target.get(group) ?? 0) + count);
   }
}

/**
 * обновление списка остатков по группам, замена значений
 */
function updateGroupList({ target, listValues }: { target: CountList; listValues: CountMap }): void {
   for (const item of target) {
      const name = item[0];
      if (listValues.has(name)) {
         item[1] = listValues.get(name) ?? item[1];
      }
   }
}

function getSum(count: CountMap): number {
   return Array.from(count.values()).reduce((sum, n) => sum + n, 0);
}

function getList<T extends string>(list: CountList<T>) {
   return list.map(e => [e[0], e[1]] as CountPair<T>).sort((a, b) => b[1] - a[1]);
}
