import React, { ReactNode } from 'react';

import { ExternalLink } from '@yandex-infracloud-ui/libs';
import { Link } from 'react-router-dom';

import { EXTERNAL_LINKS, urlBuilder } from '../models';
import { ReplicaSetFiltersParams, ReplicaSetState } from '../models/ui/yp/models';
import { TPodSet } from '../proto-typings';
import { DeployEngine } from '../models/ui/yp/view';

export function getPodSetService(
   podSet: TPodSet,
   cluster: string,
): {
   serviceId: string;
   serviceLink: ReactNode;
   deployEngine: DeployEngine;
} {
   const { meta, labels = {} } = podSet;
   const podSetId = meta?.id ?? '';

   const deployEngine = labels.deploy_engine;
   const serviceId = deployEngine === DeployEngine.YPLite ? labels.nanny_service_id : podSetId.split('.')[0];
   const serviceURL =
      deployEngine === DeployEngine.YPLite ? labels['yp.deploy_engine_url'] : urlBuilder.stage(serviceId);

   let deployEngineTitle;
   let serviceLink;
   switch (deployEngine) {
      case DeployEngine.YPLite:
         serviceLink = React.createElement(ExternalLink, { href: serviceURL }, serviceId);
         deployEngineTitle = 'YP Lite';
         break;
      case DeployEngine.DeployMCRSC:
      case DeployEngine.DeployRSC:
         serviceLink = React.createElement(Link, { to: serviceURL }, serviceId);
         deployEngineTitle = `Deploy/${deployEngine}`;
         break;
      case DeployEngine.QYP:
         serviceLink = React.createElement(ExternalLink, { href: EXTERNAL_LINKS.qypVm(cluster, podSetId) }, podSetId);
         deployEngineTitle = deployEngine;
         break;
   }

   return {
      serviceId,
      serviceLink,
      deployEngine: deployEngineTitle,
   };
}

export const formatCores = '0.[00]';

export const formatBytes = '0.0 ib';

const replicaSetStatePath: Record<ReplicaSetState, string> = {
   [ReplicaSetState.FAILED]: `[/status/failed_condition/status]="true"`,
   [ReplicaSetState.IN_PROGRESS]: `[/status/in_progress_condition/status]="true"`,
   [ReplicaSetState.READY]: `[/status/ready_condition/status]="true"`,
};

export const getReplicaSetQueryFromFilter = (filter: ReplicaSetFiltersParams): string | undefined => {
   const chunks = [];

   if (filter.replicaSetId) {
      chunks.push(`is_substr("${filter.replicaSetId}", [/meta/id])`);
   }

   if (filter.stageId) {
      chunks.push(`is_prefix('${filter.stageId}', [/meta/id])`);
   }

   if (filter.state) {
      const stateConditions = filter.state.map(state => replicaSetStatePath[state]);

      chunks.push(stateConditions.join(' OR '));
   }

   if (filter.query) {
      chunks.push(filter.query);
   }

   if (!chunks.length) {
      return undefined;
   }

   return chunks.map(x => `(${x})`).join(' AND ');
};

const multiclusterReplicaSetStatePath: Record<ReplicaSetState, string> = {
   [ReplicaSetState.FAILED]: `[/status/failed/status]="true"`,
   [ReplicaSetState.IN_PROGRESS]: `[/status/in_progress/status]="true"`,
   [ReplicaSetState.READY]: `[/status/ready/status]="true"`,
};

export const getMulticlusterReplicaSetQueryFromFilter = (filter: ReplicaSetFiltersParams): string | undefined => {
   const chunks = [];

   if (filter.replicaSetId) {
      chunks.push(`is_substr("${filter.replicaSetId}", [/meta/id])`);
   }

   if (filter.stageId) {
      chunks.push(`is_prefix('${filter.stageId}', [/meta/id])`);
   }

   if (filter.state) {
      const stateConditions = filter.state.map(state => multiclusterReplicaSetStatePath[state]);

      chunks.push(stateConditions.join(' OR '));
   }

   if (filter.query) {
      chunks.push(filter.query);
   }

   if (!chunks.length) {
      return undefined;
   }

   return chunks.map(x => `(${x})`).join(' AND ');
};
