import { faChevronDown } from '@fortawesome/free-solid-svg-icons/faChevronDown';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { Button, ButtonPin, DropdownMenu, DropdownMenuItem } from '@yandex-cloud/uikit';
import * as React from 'react';
import { SyntheticEvent } from 'react';

import { classNames } from '../../formatters';
import { getUniqueId } from '../../helpers';
import { bindAndMemoize } from '../../utils';
import { IAction } from '../models';

import styles from './ActionButtons.module.css';

interface IProps {
   actions: IAction[];
   cls?: string;
   compact: boolean;
   disabled?: boolean;
   disabledActions?: Set<string>;
   leftText?: React.ReactNode;

   onAction(e: SyntheticEvent, action: string): void;
}

const isDisabled = (props: IProps, a: IAction) =>
   props.disabled || (props.disabledActions && props.disabledActions.has(a.action));

/**
 * Панель с кнопками, часть которых может прятаться в меню.
 *
 * Т.е. есть "важные" и "неважные" кнопки. Первые видны всегда, прочие через выпадающее меню
 *
 * Наружу дергает onAction передавая id кнопки, т.н. 'action'
 */
export class ActionButtons extends React.PureComponent<IProps> {
   public static defaultProps = {
      cls: '',
      disabled: false,
      leftText: '',
   };

   private uniqueId = getUniqueId('ActionButtons');

   public render() {
      const unimportantButtons: DropdownMenuItem<void>[] = this.props.actions
         .filter(a => !a.important)
         .map(a => ({
            action: this.onAction(a.action),
            disabled: isDisabled(this.props, a),
            text: a.name,
         }));

      const hasMenu = unimportantButtons.length > 0;

      const importantButtons = this.props.actions
         .filter(a => a.important)
         .map((a, i, all) => {
            let pin: ButtonPin = i === 0 ? 'round-clear' : 'brick-clear';

            if (!hasMenu && all.length === i + 1) {
               pin = 'brick-round';
            }

            if (!hasMenu && all.length === 1) {
               pin = 'round-round';
            }

            return (
               <Button
                  key={a.action}
                  pin={pin}
                  disabled={isDisabled(this.props, a)}
                  onClick={this.onAction(a.action)}
                  className={classNames(styles.button)}
               >
                  {a.name}
               </Button>
            );
         });

      return (
         <div
            data-e2e={'ActionButtons'}
            data-id={this.uniqueId}
            className={classNames(
               {
                  [styles.compact]: this.props.compact,
               },
               styles.actionButtons,
               this.props.cls,
            )}
         >
            {!this.props.compact && <div className={styles.leftText}>{this.props.leftText}</div>}

            <div className={styles.buttons}>
               {importantButtons}

               {hasMenu && (
                  <DropdownMenu
                     items={unimportantButtons}
                     disabled={this.props.disabled}
                     switcher={
                        <Button pin={'brick-round'}>
                           <FontAwesomeIcon icon={faChevronDown} />
                        </Button>
                     }
                  />
               )}
            </div>
         </div>
      );
   }

   @bindAndMemoize
   private onAction(action: string) {
      return (e: SyntheticEvent) => {
         this.props.onAction(e, action);
      };
   }
}
