import { Dialog } from '@yandex-cloud/uikit';
import * as React from 'react';
import { SyntheticEvent } from 'react';
import { Observable } from 'rxjs';
import { takeUntil } from 'rxjs/operators';

import { toasts } from '../../toasts';
import { autobind, IDismountedProps, withDismounted } from '../../utils';
import { IActionStore, IActionWithContext, IActionWithModal, IBaseActionProps, IResult } from '../models';

import { BatchProgress } from './BatchProgress';

interface IProps {
   actionStore: IActionStore;
}

interface IState {
   actionConfig: IActionWithModal<any> | null;
   actions: Observable<any>[];
   canCloseProgress: boolean;
   context: any;
   inProgress: boolean;
   names: string[];
   visible: boolean;
}

/**
 * Специальное место для отображения модальных окон (должен быть в <App/>)
 */
class ActionContainer extends React.PureComponent<IProps & IDismountedProps, IState> {
   public static defaultProps = {};

   public static propTypes = {};

   public state: IState = {
      actionConfig: null,
      actions: [],
      canCloseProgress: false,
      context: null,
      inProgress: false,
      names: [],
      visible: false,
   };

   // FIXME как-то костыльно
   private onCancel?: () => void;

   // FIXME как-то костыльно
   private onComplete?: (results: IResult[]) => void;

   public render() {
      if (!this.state.visible) {
         return null;
      }

      const Child: React.ComponentType<IBaseActionProps<any>> | null = this.state.actionConfig
         ? this.state.actionConfig.component
         : null;

      return (
         <Dialog
            disableOutsideClick={!this.state.canCloseProgress}
            hasCloseButton={this.state.canCloseProgress}
            onClose={this.onEsc}
            open={true}
            size={'m'}
         >
            {this.state.inProgress ? (
               <BatchProgress
                  limit={5}
                  names={this.state.names}
                  actions={this.state.actions}
                  onFinish={this.onFinish}
                  canClose={this.state.canCloseProgress}
                  onClose={this.onEsc}
               />
            ) : Child ? (
               <Child
                  meta={this.state.actionConfig!}
                  context={this.state.context}
                  onRun={this.showProgress}
                  onCancel={this.closeModal}
               />
            ) : (
               'Set `component` property for action!'
            )}
         </Dialog>
      );
   }

   public componentDidMount(): void {
      for (const subject in this.props.actionStore) {
         if (this.props.actionStore.hasOwnProperty(subject)) {
            const { service } = this.props.actionStore[subject];
            service.actions.pipe(takeUntil(this.props.dismounted!)).subscribe(action => this.doAction(action));
         }
      }
   }

   private closeAll() {
      this.setState({
         actionConfig: null,
         actions: [],
         canCloseProgress: false,
         context: null,
         inProgress: false,
         visible: false,
      });
   }

   @autobind
   private closeModal() {
      if (this.state.inProgress) {
         return;
      }

      if (this.onCancel) {
         this.onCancel();
         this.onCancel = undefined;
      }
      this.onComplete = undefined;

      this.onFinish(null, []);
   }

   private doAction(action: IActionWithContext<any>): void {
      const store = this.props.actionStore[action.subject];
      const actions = store ? store.actions : [];
      const resolveNameFromContext = store ? store.resolveNameFromContext : () => [];

      const actionConfig = actions.find(a => a.action === action.action);
      if (!actionConfig) {
         toasts.error(`Unknown '${action.action} for ${action.subject}`);

         return;
      }

      this.onComplete = action.onComplete;
      this.onCancel = action.onCancel;

      this.setState({
         actionConfig,
         context: action.context,
         inProgress: false,
         names: resolveNameFromContext(action.context),
         visible: true,
      });
   }

   @autobind
   private onEsc() {
      // TODO по-хорошему, нужно проверить, если не было изменений в форме, то позволять закрывать
      if (this.state.inProgress && !this.state.canCloseProgress) {
         return;
      }

      this.closeAll();
   }

   // noinspection JSUnusedLocalSymbols
   @autobind
   private onFinish(e: SyntheticEvent | null, result: IResult[]) {
      if (this.onComplete) {
         this.onComplete(result);
         this.onComplete = undefined;
      }

      const isAllOk = result.every(r => r.success);

      if (isAllOk) {
         this.closeAll();

         return;
      }

      this.setState({ canCloseProgress: true });
   }

   // noinspection JSUnusedLocalSymbols
   @autobind
   private showProgress(e: SyntheticEvent, actions: Observable<any>[]): void {
      this.setState({
         actions,
         inProgress: true,
      });
   }
}

export const ActionContainerEnhanced = withDismounted(ActionContainer);
