import { SyntheticEvent } from 'react';
import { Observable } from 'rxjs';

import { parseApiError } from '../../http';
import { IResult } from '../models';

export enum ActionType {
   Failed = 'Failed',
   Finished = 'Finished',
   RunItem = 'RunItem',
   Success = 'Success',
   updateItem = 'updateItem',
}

interface IActionFailed {
   item: IItem;
   result: any;
   type: ActionType.Failed;
}

interface IActionFinished {
   type: ActionType.Finished;

   onFinish(e: SyntheticEvent | null, result: IResult[]): void;
}

interface IActionRunItem {
   item: IItem;
   type: ActionType.RunItem;
}

interface IActionSuccess {
   item: IItem;
   result: any;
   type: ActionType.Success;
}

interface IActionUpdateItem {
   item: IItem;
   type: ActionType.updateItem;
}

type Action = IActionFailed | IActionFinished | IActionRunItem | IActionSuccess | IActionUpdateItem;

export enum ItemState {
   enqueued = 'enqueued',
   process = 'process',
   failed = 'failed',
   success = 'success',
}

export interface IItem {
   action: Observable<any>;
   error?: string;
   name: string;
   result?: any;
   state: ItemState;
}

export const initialState = {
   actions: [] as Observable<any>[],
   errors: [] as string[],
   items: [] as IItem[],
   limit: 0,
   names: [] as string[],
   restCount: 0,
};

type State = Readonly<typeof initialState>;

export const initState = (state: State) => {
   const items = state.actions.map(
      (action, i) =>
         ({
            action,
            name: state.names[i] || `Item #${i}`,
            state: ItemState.enqueued,
         } as IItem),
   );

   return {
      ...state,
      items,
      restCount: state.actions.length,
   };
};

export const reducer = (state: State, action: Action): State => {
   switch (action.type) {
      case ActionType.Failed: {
         const item = {
            ...action.item,
            error: `${action.item.name}: ${parseApiError(action.result)}`,
            result: action.result,
            state: ItemState.failed,
         };

         return reducer(state, { type: ActionType.updateItem, item });
      }

      case ActionType.Finished: {
         const result: IResult[] = state.items.map(item => ({
            response: item.result,
            success: item.state === ItemState.success,
         }));

         action.onFinish(null, result);

         return state;
      }

      case ActionType.RunItem: {
         const items = state.items.map(item => (action.item === item ? { ...item, state: ItemState.process } : item));

         return {
            ...state,
            items,
         };
      }

      case ActionType.Success: {
         const item = {
            ...action.item,
            result: action.result,
            state: ItemState.success,
         };

         return reducer(state, { type: ActionType.updateItem, item });
      }

      case ActionType.updateItem: {
         const items = state.items.map(item => (action.item.name === item.name ? action.item : item));

         // небольшая оптимизация: пропуск пересоздания массива ошибок, если они заведомо не изменились
         const errors =
            action.item.state === ItemState.failed
               ? items.filter(item => item.state === ItemState.failed).map(item => item.error!)
               : state.errors;

         return {
            ...state,
            errors,
            items,
            restCount: state.restCount - 1,
         };
      }

      default: {
         return state;
      }
   }
};
