import { faSort } from '@fortawesome/free-solid-svg-icons/faSort';
import { Button, ButtonProps, DropdownMenu, DropdownMenuItem } from '@yandex-cloud/uikit';
import React, { useMemo } from 'react';

import { EMPTY_VALUE, EnumOption } from '../../_models';
import { FontAwesomeDataUICommonIcon } from '../FontAwesomeDataUICommonIcon/FontAwesomeDataUICommonIcon';

import classes from './SimpleSelect.module.css';

interface Props {
   disabled?: boolean;
   items: EnumOption[];
   placeholder?: string;
   qa?: string;
   switcherProps?: Omit<ButtonProps & React.RefAttributes<HTMLAnchorElement | HTMLButtonElement>, 'disabled'>;
   value: string;

   onUpdate(v: string): void;
}

/**
 * Компонент для быстрого выбора значения из ограниченного списка
 *
 * Аналог вариации <Select> из lego-on-react
 */
export const SimpleSelect: React.FC<Props> = React.memo(
   ({ disabled, onUpdate, items, value, switcherProps, placeholder = EMPTY_VALUE, qa }) => {
      const itemTitleMap = useMemo(() => new Map(items.map(item => [item.value, item.title])), [items]);

      const menuItems: DropdownMenuItem<string>[] = useMemo(
         () =>
            items.map(item => ({
               text: item.title,
               action: () => onUpdate(item.value),
            })),
         [items, onUpdate],
      );

      return (
         <DropdownMenu
            disabled={disabled}
            items={menuItems}
            switcher={
               <Button disabled={disabled} qa={qa} {...switcherProps}>
                  {value ? (
                     itemTitleMap.get(value) ?? value
                  ) : (
                     <span className={classes.placeholder}>{placeholder}</span>
                  )}
                  <FontAwesomeDataUICommonIcon className={classes.icon} icon={faSort} size={14} />
               </Button>
            }
         />
      );
   },
);

SimpleSelect.displayName = 'SimpleSelect';
