import { Checkbox } from '@yandex-cloud/uikit';
import * as React from 'react';
import { SyntheticEvent, useCallback } from 'react';
import { Link } from 'react-router-dom';

import { styleHelpers } from '../../_styles/styleHelpers';
import { classNames, forHumanCapitalized } from '../../formatters';
import { HighlightedString } from '../../small_components';

import styles from './CheckboxItem.module.css';
import { IListOption } from './models';

interface IProps {
   checked: boolean;
   cls?: string;
   hideCheckbox?: boolean;
   humanize?: boolean;
   onlyLabel?: boolean;
   option: IListOption;
   query?: string;

   getLinkHref?(v: string): string;

   onChange(e: SyntheticEvent | null, name: string): void;
}

export const CheckboxItem = React.memo(
   ({
      checked,
      cls = '',
      getLinkHref,
      hideCheckbox = false,
      humanize = false,
      onChange,
      onlyLabel = false,
      option,
      query = '',
   }: IProps) => {
      const toggle = useCallback(() => onChange(null, option.value), [onChange, option.value]);

      return (
         <Checkbox
            className={classNames(styles.checkboxItem, cls, {
               [styles.hideCheckbox]: hideCheckbox,
               [styles.onlyLabel]: onlyLabel,
               [styles.active]: hideCheckbox && checked,
            })}
            checked={checked}
            onUpdate={toggle}
            title={option.value}
         >
            <HighlightedString query={query} content={humanize ? forHumanCapitalized(option.name) : option.name} />

            {option.note ? <span className={styles.note}>({option.note})</span> : null}

            {getLinkHref ? (
               <Link to={getLinkHref(option.value)} className={classNames(styles.link, styleHelpers.linkGray)} />
            ) : null}
         </Checkbox>
      );
   },
   (prevProps, nextProps) =>
      prevProps.onChange === nextProps.onChange &&
      prevProps.checked === nextProps.checked &&
      prevProps.hideCheckbox === nextProps.hideCheckbox &&
      prevProps.option.note === nextProps.option.note &&
      prevProps.query === nextProps.query,
);

CheckboxItem.displayName = 'CheckboxItem';
