import * as React from 'react';
import {
   CSSProperties,
   forwardRef,
   Ref,
   SyntheticEvent,
   useCallback,
   useEffect,
   useImperativeHandle,
   useRef,
   useState,
} from 'react';

import '../../_polyfills/focus-options-polyfill.js';
import { classNames } from '../../formatters';
import { isEmpty, toggleSetItem } from '../../helpers';

import { CheckboxItem } from './CheckboxItem';
import styles from './CheckboxList.module.css';
import { IListOption } from './models';

interface IProps {
   cls?: string;
   humanize?: boolean;
   multiple?: boolean;
   options: IListOption[];
   style?: CSSProperties;
   useSearch?: boolean;
   value: Set<string>;

   getLinkHref?(v: string): string;

   onChange(e: SyntheticEvent, v: Set<string>): void;
}

export interface ICheckboxListHandles {
   clearQuery(): void;

   focus(): void;
}

export const CheckboxListInner = forwardRef(
   (
      {
         cls = '',
         getLinkHref,
         humanize = false,
         multiple = true,
         onChange,
         options,
         style = {},
         value,
         useSearch = true,
      }: IProps,
      ref: Ref<ICheckboxListHandles>,
   ) => {
      // hooks
      const valueRef = useRef(value);
      const checkboxListRef = useRef<HTMLUListElement>(null);
      const [query, setQuery] = useState('');
      const [filteredOptions, setFilteredOptions] = useState(options);

      const clearQuery = useCallback(() => setQuery(''), []);
      const focusList = useCallback(() => {
         requestAnimationFrame(() => {
            if (checkboxListRef.current && useSearch) {
               // the polyfill used in safari to support preventScroll option
               checkboxListRef.current.focus({ preventScroll: true });
            }
         });
      }, [useSearch]);

      useImperativeHandle(
         ref,
         () => ({
            clearQuery,
            focus: focusList,
         }),
         [clearQuery, focusList],
      );

      // effects
      // Это вынужденная мера, чтобы не пересоздавать callback toggle
      useEffect(() => {
         valueRef.current = value;
      }, [value]);

      useEffect(() => {
         const q = query.toLowerCase();
         setFilteredOptions(
            options.filter(option => [option.value, option.name].map(i => i.toLowerCase()).some(f => f.includes(q))),
         );
      }, [options, query]);

      // handlers
      const toggle = useCallback(
         (e: SyntheticEvent, name: string) => {
            const newValue = multiple ? toggleSetItem(valueRef.current, name) : new Set([name]);

            onChange(e, newValue);

            focusList();
         },
         [focusList, multiple, onChange],
      );

      const handleKeyPress = useCallback(
         (e: React.KeyboardEvent) => {
            if (!useSearch) {
               return;
            }

            const stopEvent = () => {
               e.preventDefault();
               e.nativeEvent.stopImmediatePropagation();
            };

            const { key } = e;
            switch (key) {
               case 'Escape':
                  stopEvent();
                  setQuery('');
                  break;

               case 'Backspace':
                  stopEvent();
                  setQuery(query.slice(0, query.length - 1));
                  break;

               default:
                  if (key.length === 1 && /[a-zа-яё\d -_]/i.test(key)) {
                     stopEvent();
                     setQuery(query + key);
                  }
            }
         },
         [query, useSearch],
      );

      // render
      return (
         <div className={styles.checkboxListWrapper}>
            {!useSearch || isEmpty(query) ? null : <div className={styles.search}>Search for: {query}</div>}

            <ul
               className={classNames(styles.checkboxList, cls)}
               style={style}
               onKeyDown={handleKeyPress}
               tabIndex={0}
               role={'tree'}
               ref={checkboxListRef}
            >
               {filteredOptions.length === 0 ? (
                  <span className={styles.noResults}>No search results</span>
               ) : (
                  filteredOptions.map(option => (
                     <li key={option.value} className={styles.listItem} style={style}>
                        <CheckboxItem
                           option={option}
                           checked={value.has(option.value)}
                           onChange={toggle}
                           hideCheckbox={!multiple || value.size === 0}
                           onlyLabel={!multiple}
                           humanize={humanize}
                           getLinkHref={getLinkHref}
                           query={query}
                        />
                     </li>
                  ))
               )}
            </ul>
         </div>
      );
   },
);

CheckboxListInner.displayName = 'CheckboxListInner';

export const CheckboxList = React.memo(CheckboxListInner);

CheckboxList.displayName = 'CheckboxList';
