import { faAngleDown } from '@fortawesome/free-solid-svg-icons/faAngleDown';
import { faAngleRight } from '@fortawesome/free-solid-svg-icons/faAngleRight';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { Button } from '@yandex-cloud/uikit';
import * as React from 'react';
import { SyntheticEvent } from 'react';

import { toggleSetItem } from '../../helpers';
import { autobind } from '../../utils';
import { CheckboxItem } from '../CheckboxList/CheckboxItem';

import styles from './CheckboxTree.module.css';
import { getExpanded, ITreeOption } from './models';

interface IProps {
   cls?: string;
   humanize?: boolean;
   options: ITreeOption[];
   value: Set<string>;

   onChange(e: SyntheticEvent, v: Set<string>): void;
}

interface IState {
   expanded: Set<string>;
}

export class CheckboxTree extends React.PureComponent<IProps, IState> {
   public static defaultProps = {
      humanize: false,
   };

   constructor(props: IProps) {
      super(props);

      this.state = {
         expanded: getExpanded(this.props.options, this.props.value),
      };
   }

   public render() {
      return <div className={this.props.cls}>{this.renderBranch(this.props.options)}</div>;
   }

   public renderBranch(options: ITreeOption[]) {
      return <ul className={styles.checkboxTree}>{options.map(o => this.renderItem(o))}</ul>;
   }

   @autobind
   private onToggle(e: SyntheticEvent, name: string) {
      const newValue = toggleSetItem(this.props.value, name);

      this.props.onChange(e, newValue);
   }

   @autobind
   private onToggleExpand(id: string) {
      this.setState(s => ({
         expanded: toggleSetItem(s.expanded, id),
      }));
   }

   private renderItem(option: ITreeOption) {
      const hasChildren = option.children && option.children.length > 0;
      const isExpanded = this.state.expanded.has(option.value);
      const isChecked = this.props.value.has(option.value);

      return (
         <li className={styles.listItem} key={option.value}>
            <div className={styles.listItemContent}>
               {hasChildren ? (
                  <Button view={'flat'} size={'s'} onClick={() => this.onToggleExpand(option.value)}>
                     <FontAwesomeIcon icon={isExpanded ? faAngleDown : faAngleRight} fixedWidth={true} />
                  </Button>
               ) : (
                  <div className={styles.buttonStub} />
               )}

               <CheckboxItem
                  cls={styles.checkboxItem}
                  option={option}
                  checked={isChecked}
                  onChange={this.onToggle}
                  humanize={this.props.humanize}
               />
            </div>

            {hasChildren && isExpanded && this.renderBranch(option.children!)}
         </li>
      );
   }
}
