import { Button, Popup, TextInput, TextInputSize } from '@yandex-cloud/uikit';
import endOfDay from 'date-fns/endOfDay';
import getDate from 'date-fns/getDate';
import getHours from 'date-fns/getHours';
import getMinutes from 'date-fns/getMinutes';
import getMonth from 'date-fns/getMonth';
import getSeconds from 'date-fns/getSeconds';
import getYear from 'date-fns/getYear';
import isValid from 'date-fns/isValid';
import setHours from 'date-fns/setHours';
import setMinutes from 'date-fns/setMinutes';
import setSeconds from 'date-fns/setSeconds';
import startOfDay from 'date-fns/startOfDay';
import * as React from 'react';

import {
   DATEPICKER_FORMAT,
   IValueProps,
   Keys,
   NullableDate,
   TIMEPICKER_FORMAT,
   TIMEPICKER_FORMAT_SEC,
} from '../../_models';
import { styleHelpers } from '../../_styles/styleHelpers';
import { formatDate } from '../../formatters';
import { isEqual, parseDate } from '../../helpers';
import { autobind } from '../../utils';

import calendarSvg from './calendar.svg';
import { Calendar } from './components/Calendar';
import { TimeInput } from './components/TimeInput';
import styles from './DateTimePicker.module.css';

export enum DefaultTime {
   StartOfDay,
   EndOfDay,
   Current,
}

interface IProps extends IValueProps<NullableDate> {
   cls?: string;
   defaultTimeMode?: DefaultTime;
   disabled?: boolean;
   id?: string;
   name?: string;
   placeholder?: string;
   qa?: string;
   showTimeString?: boolean;
   size?: TextInputSize;
   timePlaceholder?: string;
   withSeconds?: boolean;
   withTime?: boolean;

   onBlur?(e: { target: { name: string } }): void;
}

interface IState {
   date: Date;
   dateString: string;
   opened: boolean;
   timeString: string;
}

export class DateTimePicker extends React.PureComponent<IProps, IState> {
   public static defaultProps = {
      cls: '',
      disabled: false,
      placeholder: DATEPICKER_FORMAT,
      showTimeString: false,
      size: 'm',
      withSeconds: false,
      withTime: false,
   };

   private ref = React.createRef<HTMLDivElement>();

   private readonly timePickerFormat: string;

   constructor(props: IProps) {
      super(props);
      const { withSeconds } = props;

      this.timePickerFormat = withSeconds ? TIMEPICKER_FORMAT_SEC : TIMEPICKER_FORMAT;

      this.state = {
         date: this.props.value ?? this.getDefaultDate(),
         dateString: this.formatDate(this.props.value),
         opened: false,
         timeString: this.formatTime(this.props.value),
      };
   }

   public render() {
      const { cls, disabled, placeholder, showTimeString, size, timePlaceholder, withTime, withSeconds } = this.props;
      const { date, dateString, timeString, opened } = this.state;

      return (
         <div className={cls}>
            <div className={styles.inputWrapper} ref={this.ref}>
               {withTime && showTimeString ? (
                  <div className={withSeconds ? styles.timeInputWithSeconds : styles.timeInput}>
                     <TextInput
                        disabled={disabled}
                        onBlur={this.onTimeInputBlur}
                        onFocus={this.show}
                        onUpdate={this.onTimeInputChange}
                        pin={'round-clear'}
                        placeholder={timePlaceholder ?? this.timePickerFormat}
                        size={size}
                        value={timeString}
                        qa={this.props.qa ? `${this.props.qa}:Time` : undefined}
                     />
                  </div>
               ) : null}

               <TextInput
                  pin={withTime && showTimeString ? 'brick-round' : 'round-round'}
                  size={size}
                  value={dateString}
                  disabled={disabled}
                  name={this.props.name}
                  id={this.props.id}
                  onUpdate={this.onDateInputChange}
                  onKeyDown={this.onKeyDown}
                  onFocus={this.show}
                  onBlur={this.onDateInputBlur}
                  placeholder={placeholder}
                  className={styles.input}
                  qa={this.props.qa ? `${this.props.qa}:Date` : undefined}
               />
               {/* eslint-disable-next-line jsx-a11y/no-noninteractive-element-to-interactive-role */}
               <img alt={'cal'} src={calendarSvg} className={styles.icon} role={'button'} tabIndex={0} />
            </div>

            <Popup anchorRef={this.ref} hasArrow={true} onClose={this.close} open={opened} className={styles.popup}>
               <Calendar value={date} onChange={this.onDateChange} />

               {withTime ? (
                  <div className={styles.footer}>
                     <TimeInput value={date} onChange={this.onTimeChange} showSeconds={withSeconds} />
                     <Button view={'action'} className={styleHelpers.leftSpace} onClick={this.close}>
                        Select
                     </Button>
                  </div>
               ) : null}
            </Popup>
         </div>
      );
   }

   public componentDidUpdate(prevProps: Readonly<IProps>): void {
      if (!isEqual(this.props.value, prevProps.value)) {
         this.setState({
            date: this.props.value ?? this.getDefaultDate(),
            dateString: this.formatDate(this.props.value),
            timeString: this.formatTime(this.props.value),
         });
      }
   }

   @autobind
   private close() {
      this.setState({ opened: false });

      if (this.props.onBlur) {
         this.props.onBlur({ target: { name: this.props.name! } });
      }
   }

   private formatDate(v: NullableDate): string {
      return v === null ? '' : formatDate(v, DATEPICKER_FORMAT);
   }

   private formatTime(v: NullableDate): string {
      return v === null ? '' : formatDate(v, this.timePickerFormat);
   }

   private getDefaultDate(): Date {
      const now = new Date();

      switch (this.props.defaultTimeMode) {
         case DefaultTime.Current:
            return now;

         case DefaultTime.StartOfDay:
            return startOfDay(now);

         case DefaultTime.EndOfDay:
            return endOfDay(now);

         default:
            return this.props.withTime ? now : startOfDay(now);
      }
   }

   // noinspection JSUnusedLocalSymbols
   /**
    * Обработчик изменения даты
    *
    * Важно: дата приходит с временем в начале дня. Т.е. время всегда одно и то же.
    */
   @autobind
   private onDateChange(e: React.SyntheticEvent | null, v: Date) {
      const { date } = this.state;
      const [h, min, sec] = [getHours(date), getMinutes(date), getSeconds(date)];
      const [y, m, d] = [getYear(v), getMonth(v), getDate(v)];

      this.updateValue(new Date(y, m, d, h, min, sec));
   }

   @autobind
   private onDateInputBlur(e: React.FocusEvent<HTMLInputElement>): void {
      this.parseAndChangeDate(e.target.value);

      if (this.props.onBlur) {
         this.props.onBlur({ target: { name: this.props.name! } });
      }
   }

   @autobind
   private onDateInputChange(value: string): void {
      this.setState({ dateString: value });
   }

   @autobind
   private onKeyDown(e: React.KeyboardEvent<HTMLInputElement>) {
      if (e.key === Keys.Enter) {
         this.parseAndChangeDate(this.state.dateString);
         this.close();
      }
   }

   // noinspection JSUnusedLocalSymbols
   /**
    * Обработчик изменения времени (только часов-минут)
    */
   @autobind
   private onTimeChange(e: React.SyntheticEvent | null, v: Date) {
      const { date } = this.state;
      const [h, min, sec] = [getHours(v), getMinutes(v), getSeconds(v)];
      const [y, m, d] = [getYear(date), getMonth(date), getDate(date)];

      this.updateValue(new Date(y, m, d, h, min, sec));
   }

   @autobind
   private onTimeInputBlur(e: React.FocusEvent<HTMLInputElement>): void {
      const parts = e.target.value.split(':');
      let date = new Date();

      const [h, m] = parts;
      date = setHours(date, parseInt(h, 10));
      date = setMinutes(date, parseInt(m, 10));

      if (this.props.withSeconds && parts[2]) {
         const s = parts[2];
         date = setSeconds(date, parseInt(s, 10));
      } else {
         date = setSeconds(date, 0);
      }

      if (isValid(date)) {
         this.onTimeChange(null, date);
      } else {
         this.props.onChange(e, null);
      }

      if (this.props.onBlur) {
         this.props.onBlur({ target: { name: this.props.name! } });
      }
   }

   @autobind
   private onTimeInputChange(value: string): void {
      this.setState({ timeString: value });
   }

   private parseAndChangeDate(d: string) {
      const date = parseDate(d);

      if (!isValid(date)) {
         this.props.onChange(null, null);

         return;
      }

      this.onDateChange(null, date);
   }

   @autobind
   private show() {
      if (!this.state.opened) {
         this.setState({ opened: true });
      }
   }

   private updateValue(v: Date): void {
      this.setState(
         {
            date: v,
            dateString: formatDate(v, DATEPICKER_FORMAT),
            timeString: formatDate(v, this.timePickerFormat),
         },
         () => this.props.onChange(null, v),
      );
   }
}
