import { faAngleLeft } from '@fortawesome/free-solid-svg-icons/faAngleLeft';
import { faAngleRight } from '@fortawesome/free-solid-svg-icons/faAngleRight';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { Button } from '@yandex-cloud/uikit';
import addDays from 'date-fns/addDays';
import addMonths from 'date-fns/addMonths';
import format from 'date-fns/format';
import getYear from 'date-fns/getYear';
import isMonday from 'date-fns/isMonday';
import isSameDay from 'date-fns/isSameDay';
import isSameMonth from 'date-fns/isSameMonth';
import isWeekend from 'date-fns/isWeekend';
import startOfDay from 'date-fns/startOfDay';
import startOfMonth from 'date-fns/startOfMonth';
import * as React from 'react';
import { SyntheticEvent } from 'react';

import { classNames } from '../../../formatters';
import { isEqual } from '../../../helpers';
import { autobind } from '../../../utils';

import styles from './Calendar.module.css';
import { Day } from './models';

function getMonthName(date: Date): string {
   const isCurrentYear = getYear(new Date()) === getYear(date);

   return format(date, isCurrentYear ? 'MMMM' : 'MMMM yyyy');
}

export function getMonday(date: Date): Date {
   while (!isMonday(date)) {
      date = addDays(date, -1);
   }

   return date;
}

interface IProps {
   value: Date;

   onChange(e: SyntheticEvent, v: Date): void;
}

interface IState {
   currentMonth: Date;
}

export class Calendar extends React.PureComponent<IProps, IState> {
   constructor(props: IProps) {
      super(props);

      this.state = {
         currentMonth: startOfMonth(this.props.value),
      };
   }

   public componentDidUpdate(prevProps: Readonly<IProps>): void {
      if (!isEqual(this.props.value, prevProps.value)) {
         this.setState({ currentMonth: startOfMonth(this.props.value) });
      }
   }

   public render() {
      const start = getMonday(this.state.currentMonth);
      const end = addDays(start, 7 * 6 - 1); // 6 weeks
      const days: Day[] = [];
      for (let date = start; date <= end; date = addDays(date, 1)) {
         days.push(new Day(date));
      }

      // Подписи к колонкам с днями недели
      const weekDays = days.slice(0, 7).map(d => {
         const label = format(d.value, 'iiiiii');

         return (
            <span key={label} className={isWeekend(d.value) ? styles.weekendLabel : styles.label}>
               {label}
            </span>
         );
      });

      const monthSwitcher = (
         <div className={styles.month}>
            <Button view={'flat'} onClick={this.prevMonth}>
               <FontAwesomeIcon icon={faAngleLeft} />
            </Button>
            <div className={styles.month_name}>{getMonthName(this.state.currentMonth)}</div>
            <Button view={'flat'} onClick={this.nextMonth}>
               <FontAwesomeIcon icon={faAngleRight} />
            </Button>
         </div>
      );

      return (
         <div className={styles.calendar}>
            {monthSwitcher}

            <div className={styles.days}>
               {weekDays}

               {days.map(d => (
                  <span
                     key={d.key}
                     className={classNames(
                        {
                           [styles.weekend]: d.isWeekend,
                           [styles.selected]: isSameDay(d.value, this.props.value),
                           [styles.other_month]: !isSameMonth(d.value, this.state.currentMonth),
                        },
                        styles.day,
                     )}
                     onClick={this.onDayClick(d.value)}
                     role={'button'}
                     tabIndex={0}
                  >
                     {d.day}
                  </span>
               ))}
            </div>
         </div>
      );
   }

   @autobind
   private nextMonth() {
      this.setState(state => ({
         currentMonth: addMonths(state.currentMonth, 1),
      }));
   }

   @autobind
   private onDayClick(value: Date) {
      return (e: SyntheticEvent) => {
         this.props.onChange(e, startOfDay(value));
         this.setState({ currentMonth: startOfMonth(value) });
      };
   }

   @autobind
   private prevMonth() {
      this.setState(state => ({
         currentMonth: addMonths(state.currentMonth, -1),
      }));
   }
}
