import * as React from 'react';
import { ReactNode, SyntheticEvent, useCallback, useRef } from 'react';

import { classNames } from '../../formatters';
import { Dropdown, DropdownRef } from '../../small_components';
import { CheckboxList, ICheckboxListHandles } from '../CheckboxList/CheckboxList';
import { IListOption } from '../CheckboxList/models';

import classes from './DropdownCheckboxList.module.css';

interface IProps {
   className?: string;
   disabled?: boolean;
   greyPlaceholder?: boolean;
   humanize?: boolean;
   limitPopupWidth?: boolean;
   multiple?: boolean;
   name?: string;
   options: IListOption[];
   placeholder?: string;
   value: Set<string>;

   getLabel?(count: number, v?: Set<string>): ReactNode | string;

   onChange(e: SyntheticEvent, v: Set<string>): void;
}

export const DropdownCheckboxList = React.memo(
   ({
      className,
      disabled = false,
      getLabel,
      humanize = false,
      multiple = true,
      greyPlaceholder = false,
      limitPopupWidth = true,
      name,
      onChange,
      options,
      placeholder,
      value,
   }: IProps) => {
      // hooks
      const dropdownRef = useRef<DropdownRef>(null);
      const checkboxListRef = useRef<ICheckboxListHandles>(null);

      // handlers
      const handleToggling = useCallback((visible: boolean) => {
         if (checkboxListRef.current && visible) {
            checkboxListRef.current.focus();
         }

         if (checkboxListRef.current && !visible) {
            checkboxListRef.current.clearQuery();
         }
      }, []);

      const close = useCallback(() => {
         if (dropdownRef.current) {
            dropdownRef.current.close();
         }
      }, []);

      const handleChange = useCallback(
         (e: SyntheticEvent, v: Set<string>) => {
            onChange(e, v);

            if (!multiple) {
               close();
            }
         },
         [close, multiple, onChange],
      );

      // render

      const getButtonText = useCallback((): ReactNode => {
         if (getLabel) {
            return getLabel(value.size, value);
         }

         return (
            Array.from(value)
               .map(v => options.find(o => o.value === v))
               .filter(Boolean)
               .map(o => o!.name)
               .join(', ') || placeholder
         );
      }, [getLabel, options, placeholder, value]);

      const switcher = (
         <div
            className={classNames({
               [classes.empty]: greyPlaceholder && value.size === 0,
            })}
         >
            {getButtonText()}
         </div>
      );

      return (
         <Dropdown
            className={className}
            disabled={disabled}
            limitPopupWidth={limitPopupWidth}
            onToggle={handleToggling}
            ref={dropdownRef}
            switcher={switcher}
            testE2e={name}
         >
            <CheckboxList
               ref={checkboxListRef}
               options={options}
               onChange={handleChange}
               value={value}
               humanize={humanize}
               multiple={multiple}
            />
         </Dropdown>
      );
   },
);

DropdownCheckboxList.displayName = 'DropdownCheckboxList';
