import { Checkbox, TextInput } from '@yandex-cloud/uikit';
import React, { useCallback, useEffect, useRef, useState } from 'react';

import { BaseInputProps, EMPTY_VALUE, OptionalNumber } from '../../_models';
import { styleHelpers } from '../../_styles/styleHelpers';
import { classNames, formatNumber } from '../../formatters';
import { isEmpty } from '../../helpers';
import { usePreventScrollForInputs } from '../../react_hooks';

import classes from './OptionalNumberInput.module.css';

function getInternalValue(value: OptionalNumber) {
   return typeof value !== 'number' || Number.isNaN(value) ? '' : value.toString();
}

interface Props extends BaseInputProps<OptionalNumber> {
   max?: number;
   min?: number;
}

export const OptionalNumberInput: React.FC<Props> = React.memo(
   ({
      children,
      className,
      disabled = false,
      hasError,
      id,
      max,
      min = 0,
      name,
      onBlur,
      onChange,
      placeholder,
      qa,
      readonly = false,
      readonlyClassName,
      value,
   }) => {
      const inputRef = useRef<HTMLInputElement>();

      const [checked, setChecked] = useState(!isEmpty(value));

      const [internalValue, setInternalValue] = useState(getInternalValue(value));

      useEffect(() => {
         setInternalValue(getInternalValue(value));

         if (isEmpty(value)) {
            setChecked(false);
         }
      }, [value]);

      // region handlers
      const toggle = useCallback(() => {
         setChecked(!checked);

         onChange(checked ? null : 1);

         if (!checked) {
            setTimeout(() => inputRef.current?.focus(), 50);
         }

         if (onBlur) {
            onBlur();
         }
      }, [checked, onBlur, onChange]);

      const handleBlur = useCallback(() => {
         if (inputRef.current) {
            if (isEmpty(internalValue)) {
               onChange(null);
            } else {
               const parsed = parseInt(internalValue, 10);
               onChange(parsed > min && (max === undefined ? true : parsed <= max) ? parsed : null);
            }
         }

         if (onBlur) {
            onBlur();
         }
      }, [internalValue, max, min, onBlur, onChange]);

      const handleLabelClick = useCallback(() => {
         if (disabled) {
            return;
         }

         if (checked) {
            inputRef.current?.focus();
         } else {
            toggle();
         }
      }, [checked, disabled, toggle]);

      const { setInputRef: setInputRefForPreventScrolling } = usePreventScrollForInputs();

      const setInputRef = useCallback(
         (el: HTMLInputElement) => {
            inputRef.current = el;
            setInputRefForPreventScrolling(el);
         },
         [setInputRefForPreventScrolling],
      );
      // endregion

      // region render
      if (readonly) {
         return (
            <div className={classNames(classes.wrapper, readonlyClassName, className)}>
               {children}: {value === null || value === undefined ? EMPTY_VALUE : formatNumber(value)}
            </div>
         );
      }

      // TODO разобраться с label
      return (
         <div className={classNames(classes.wrapper, className, { [classes.disabled]: disabled })}>
            <Checkbox
               checked={checked}
               className={classes.checkbox}
               disabled={disabled}
               onChange={toggle}
               qa={qa ? `${qa}:Check` : undefined}
            />
            {/* eslint-disable-next-line jsx-a11y/no-static-element-interactions */}
            <span className={classNames(classes.label, { [classes.error]: hasError })} onClick={handleLabelClick}>
               {children}
            </span>
            <TextInput
               className={classNames(classes.input, { [styleHelpers.commonError]: hasError })}
               controlProps={{ min, max }}
               controlRef={setInputRef}
               disabled={disabled || !checked}
               id={id}
               name={name}
               onBlur={handleBlur}
               onUpdate={setInternalValue}
               placeholder={placeholder}
               qa={qa}
               type={'number'}
               value={internalValue ?? ''}
            />
         </div>
      );
      // endregion
   },
);

OptionalNumberInput.displayName = 'OptionalNumberInput';
