import { TextInput } from '@yandex-cloud/uikit';
import React, { ReactNode, useCallback, useEffect, useMemo, useState } from 'react';

import { BaseInputProps, EMPTY_VALUE, OptionalNumber } from '../../_models';
import { SimpleSelect } from '../../data-ui-common-extra';
import { classNames, formatNumber } from '../../formatters';
import { isEmpty } from '../../helpers';
import { usePreventScrollForInputs } from '../../react_hooks';

import classes from './RoundedInput.module.css';
import { getInitValues } from './utils';

export interface RoundMeasure {
   multiplier: number;
   name: string;
}

interface Props extends BaseInputProps<OptionalNumber> {
   defaultMeasure: string;
   fixedWidth?: number;
   measures: RoundMeasure[];

   useRenderDetails?: boolean;

   renderDetails?(value: number): ReactNode;
}

export const RoundedInput: React.FC<Props> = React.memo(
   ({
      className,
      defaultMeasure: defaultMeasureName,
      disabled,
      fixedWidth,
      measures,
      name,
      onBlur,
      onChange,
      placeholder,
      qa,
      readonly,
      readonlyClassName,
      renderDetails,
      useRenderDetails = false,
      value,
   }) => {
      const defaultMeasure = measures.find(m => m.name === defaultMeasureName);
      if (!defaultMeasure) {
         throw new Error(
            `Unknown defaultMeasure "${defaultMeasureName}", known are: ${measures.map(m => m.name).join(', ')}`,
         );
      }

      if (useRenderDetails && !renderDetails) {
         throw new Error('You should define renderDetails function if useRenderDetails=true');
      }

      const baseMeasure = measures.find(m => m.multiplier === 1);
      if (!baseMeasure) {
         throw new Error(`You should define base measure (with multiplier = 1)`);
      }

      const [initNumberValue, initMeasure] = useMemo(
         () => getInitValues(value, measures, defaultMeasure),
         [defaultMeasure, measures, value],
      );
      const [numberValue, setNumberValue] = useState(initNumberValue);
      const [measure, setMeasure] = useState(initMeasure);

      useEffect(() => {
         if (isEmpty(value)) {
            setNumberValue(null);

            return;
         }

         // Отработка изменения значения из-вне
         const measureConfig = measures.find(i => i.name === measure) ?? defaultMeasure;
         const newValue = (value ?? 0) / measureConfig.multiplier;

         // Можно отобразить в текущей measure (целое число)
         if (newValue % 1 === 0) {
            setNumberValue(newValue);

            return;
         }

         // Придётся конвертировать
         const [v, m] = getInitValues(value, measures, defaultMeasure);
         setNumberValue(v);
         setMeasure(m);
      }, [defaultMeasure, measure, measures, value]);

      const handleNumberChange = useCallback(
         (v: string) => {
            if (isEmpty(v)) {
               setNumberValue(null);
               onChange(null);

               return;
            }

            const vInt = parseInt(v, 10);

            setNumberValue(vInt);

            const measureConfig = measures.find(m => m.name === measure) ?? defaultMeasure;
            onChange(vInt * measureConfig.multiplier);
         },
         [defaultMeasure, measure, measures, onChange],
      );

      const handleMeasureChange = useCallback(
         (v: string) => {
            const measureConfig = measures.find(m => m.name === v);
            if (measureConfig) {
               setMeasure(measureConfig.name);

               if (numberValue !== null && numberValue !== undefined) {
                  onChange(numberValue * measureConfig.multiplier);
               }

               if (onBlur) {
                  onBlur();
               }
            }
         },
         [measures, numberValue, onBlur, onChange],
      );

      const { setInputRef } = usePreventScrollForInputs();

      const wrapperClasses = classNames(classes.wrapper, className);
      if (readonly) {
         return (
            <div className={wrapperClasses}>
               <div className={readonlyClassName}>
                  {value !== null && value !== undefined
                     ? renderDetails
                        ? renderDetails(value)
                        : formatNumber(value)
                     : EMPTY_VALUE}
                  <input type={'hidden'} name={name} value={value?.toString()} data-qa={qa} />
               </div>
            </div>
         );
      }

      return (
         <div className={wrapperClasses}>
            <TextInput
               className={classes.input}
               controlProps={{ min: 0 }}
               disabled={disabled}
               name={`${name}_number`}
               onBlur={onBlur}
               onUpdate={handleNumberChange}
               pin={'round-clear'}
               placeholder={placeholder}
               value={numberValue === null || numberValue === undefined ? '' : numberValue.toString()}
               type={'number'}
               ref={setInputRef}
               qa={qa ? `${qa}:Value` : undefined}
            />

            <SimpleSelect
               items={measures.map(m => ({ value: m.name, title: m.name }))}
               disabled={disabled}
               switcherProps={{
                  pin: 'clear-round',
                  style: { minWidth: fixedWidth },
               }}
               value={measure}
               onUpdate={handleMeasureChange}
               qa={qa ? `${qa}:Measure` : undefined}
            />

            {useRenderDetails && renderDetails && value !== null && value !== undefined ? (
               <div className={classes.details}>{renderDetails(value)}</div>
            ) : null}
         </div>
      );
   },
);

RoundedInput.displayName = 'RoundedInput';
