import React, { useCallback, useEffect, useRef, useState } from 'react';

import { BaseInputProps, EMPTY_VALUE } from '../../_models';
import { styleHelpers } from '../../_styles/styleHelpers';
import { classNames } from '../../formatters';
import { arrayToSet } from '../../helpers';
import { Rows } from '../../small_components';
import { AutoGrowTextArea } from '../AutoGrowTextArea/AutoGrowTextArea';

const DEFAULT_SEPARATOR = /[,\s]/;

function splitString(v: string, separator?: RegExp): string[] {
   return v.split(separator || DEFAULT_SEPARATOR).filter(r => r.trim());
}

interface Props extends BaseInputProps<Set<string> | Set<number>> {
   separator?: RegExp;
   type?: 'string' | 'number';
}

export const SetValuesInput: React.FC<Props> = React.memo(
   ({
      className,
      disabled,
      hasError,
      id,
      name,
      onBlur,
      onChange,
      placeholder,
      qa,
      readonly,
      readonlyClassName,
      separator = DEFAULT_SEPARATOR,
      type = 'string',
      value,
   }) => {
      const focused = useRef(false);
      const [text, setText] = useState<string>(Array.from(value as Set<string | number>).join('\n'));

      const updateInternalValue = useCallback(() => {
         if (focused.current) {
            return;
         }

         setText(Array.from(value as Set<string | number>).join('\n'));
      }, [value]);

      useEffect(() => updateInternalValue(), [updateInternalValue, value]);

      // region handlers
      const update = useCallback(
         (v: string) => {
            setText(v);

            switch (type) {
               case 'string': {
                  const lines = splitString(v, separator);

                  onChange(arrayToSet(lines));
                  break;
               }

               case 'number': {
                  const numbers = splitString(v, separator)
                     .map(n => parseInt(n, 10))
                     .filter(n => !Number.isNaN(n));

                  onChange(arrayToSet(numbers));
                  break;
               }
               default:
                  break;
            }
         },
         [onChange, separator, type],
      );

      const onFocus = useCallback(() => {
         focused.current = true;
      }, []);

      const handleBlur = useCallback(() => {
         focused.current = false;
         updateInternalValue();

         if (onBlur) {
            onBlur();
         }
      }, [onBlur, updateInternalValue]);
      // endregion

      // region render
      if (readonly) {
         return (
            <div className={readonlyClassName}>
               <Rows text={text || EMPTY_VALUE} />
               <input type={'hidden'} name={name} value={text} data-qa={qa} />
            </div>
         );
      }

      return (
         <AutoGrowTextArea
            className={classNames(className, { [styleHelpers.commonError]: hasError })}
            disabled={disabled}
            hasClear={true}
            id={id}
            name={name}
            onBlur={handleBlur}
            onFocus={onFocus}
            onUpdate={update}
            placeholder={placeholder}
            value={text}
            qa={qa}
         />
      );
      // endregion
   },
);

SetValuesInput.displayName = 'SetValuesInput';
