import React, { useCallback, useMemo } from 'react';

import { SimpleSelect } from '../../data-ui-common-extra';

interface TimeZone {
   // международный код таймзоны
   key: string;

   // разница относительно московского времени
   mskOffset: number;

   // отображаемый заголовок по московского времени
   mskTitle: string;

   // разница относительно гринвича
   utcOffset: number;

   // отображаемый заголовок по UTC
   utcTitle: string;
}

export const TIME_ZONES: TimeZone[] = [
   { key: 'Europe/Kaliningrad', mskOffset: 60, utcOffset: -120, mskTitle: 'MSK-1', utcTitle: 'UTC+2' },
   { key: 'Europe/Moscow', mskOffset: 0, utcOffset: -180, mskTitle: 'MSK', utcTitle: 'UTC+3' },
   { key: 'Europe/Samara', mskOffset: -60, utcOffset: -240, mskTitle: 'MSK+1', utcTitle: 'UTC+4' },
   { key: 'Asia/Yekaterinburg', mskOffset: -120, utcOffset: -300, mskTitle: 'MSK+2', utcTitle: 'UTC+5' },
   { key: 'Asia/Omsk', mskOffset: -180, utcOffset: -360, mskTitle: 'MSK+3', utcTitle: 'UTC+6' },
   { key: 'Asia/Krasnoyarsk', mskOffset: -240, utcOffset: -420, mskTitle: 'MSK+4', utcTitle: 'UTC+7' },
   { key: 'Asia/Irkutsk', mskOffset: -300, utcOffset: -480, mskTitle: 'MSK+5', utcTitle: 'UTC+8' },
   { key: 'Asia/Yakutsk', mskOffset: -360, utcOffset: -540, mskTitle: 'MSK+6', utcTitle: 'UTC+9' },
   { key: 'Asia/Vladivostok', mskOffset: -420, utcOffset: -600, mskTitle: 'MSK+7', utcTitle: 'UTC+10' },
   { key: 'Asia/Magadan', mskOffset: -480, utcOffset: -660, mskTitle: 'MSK+8', utcTitle: 'UTC+11' },
   { key: 'Asia/Kamchatka', mskOffset: -540, utcOffset: -720, mskTitle: 'MSK+9', utcTitle: 'UTC+12' },
];

interface Props {
   displayMode?: 'MSK' | 'UTC';
   timeZoneKey?: string;
   timeZones?: TimeZone[];

   onChange(timeZone: TimeZone): void;
}

export const TimeZonePicker: React.FC<Props> = React.memo(
   ({ displayMode = 'UTC', onChange, timeZoneKey, timeZones = TIME_ZONES }) => {
      const onSelect = useCallback(
         (key: string) => {
            const timeZoneData = timeZones.find(x => x.key === key);

            if (timeZoneData) {
               onChange(timeZoneData);
            }
         },
         [onChange, timeZones],
      );

      const items = useMemo(
         () =>
            timeZones.map(z => ({
               value: z.key,
               title: displayMode === 'MSK' ? z.mskTitle : z.utcTitle,
            })),
         [displayMode, timeZones],
      );

      return <SimpleSelect items={items} value={timeZoneKey ?? ''} onUpdate={onSelect} />;
   },
);

TimeZonePicker.displayName = 'TimeZonePicker';
