import { storiesOf } from '@storybook/react';
import { Checkbox } from '@yandex-cloud/uikit';
import * as React from 'react';
import { SyntheticEvent, useState } from 'react';
import { object, string } from 'yup';

import { NullableDate } from '../../_models';
import { getValidResult, IValidationResult, validateEntity } from '../../utils';
import { FormFieldType, IFormField } from '../models';

import { FormLayout } from './FormLayout';

interface IParams {
   age: number;
   bio: string;
   birthday: NullableDate;
   carNumber: string;
   children: string;
   hasCar: boolean;
   hasChildren: boolean;
   name: string;
   sex: 'male' | 'female';
   spouse: string;
}

const paramsShape = {
   bio: string().max(140),
};
const schema = object(paramsShape);

const fields: IFormField<IParams>[] = [
   {
      getError: () => 'test error',
      hint: 'Hint for name',
      label: 'Full name',
      name: 'name',
      placeholder: 'Full name placeholder',
      required: true,
      type: FormFieldType.String,
   },
   {
      getError: p => (p.sex === 'male' ? 'test error' : null),
      getHelp: () => 'Hint for sex',
      hint: 'Hint for sex',
      label: 'Sex sex sex',
      name: 'sex',
      options: [
         { title: 'Male', value: 'male' },
         { title: 'Female', value: 'female' },
      ],
      type: FormFieldType.Enum,
   },
   {
      getError: () => 'test error',
      label: 'Birthday',
      name: 'birthday',
      type: FormFieldType.Date,
   },
   {
      getError: () => 'test error',
      label: 'Age',
      name: 'age',
      type: FormFieldType.Number,
   },
   {
      getHelp: p => (p.bio ? p.bio : 'Hint for bio'),
      label: 'Biography',
      name: 'bio',
      placeholder: 'Short (140 symbols maximum) description of your life (edit help)',
      type: FormFieldType.Text,
   },
   {
      label: 'Spouse',
      name: 'spouse',
      type: FormFieldType.String,
   },
   {
      checkboxLabel: 'Has children',
      getError: () => 'test error',
      getHelp: () => 'Has any child (should switch disabled status of Names of children field)',
      label: '',
      name: 'hasChildren',
      type: FormFieldType.BooleanSwitcher,
   },
   {
      isHidden: p => !p.hasChildren,
      label: 'Names of children',
      name: 'children',
      type: FormFieldType.StringSet,
   },
   {
      checkboxLabel: 'Has car',
      getHelp: () => 'Has any car (should hide car number field if false)',
      label: '',
      name: 'hasCar',
      type: FormFieldType.Boolean,
   },
   {
      isHidden: p => !p.hasCar,
      label: 'Car number',
      name: 'carNumber',
      type: FormFieldType.String,
   },
];

const FormLayoutStory = () => {
   // hooks
   const [readonly, setReadonly] = useState(false);
   const [params, setParams] = useState<IParams>({
      age: 0,
      bio: '',
      birthday: null,
      carNumber: '',
      children: '',
      hasCar: false,
      hasChildren: false,
      name: '',
      sex: 'male',
      spouse: '',
   });
   const [validation, setValidation] = useState<IValidationResult<IParams>>(getValidResult());

   // handlers
   const onChange = (e: SyntheticEvent, p: IParams) => {
      const vr = validateEntity(schema, Object.keys(paramsShape) as any[], p, null);

      setParams(p);
      setValidation(vr);
   };
   const toggleReadonly = () => setReadonly(r => !r);

   // render
   return (
      <>
         <h1>
            <Checkbox checked={readonly} onUpdate={toggleReadonly}>
               Readonly
            </Checkbox>
         </h1>

         <FormLayout
            readonly={readonly}
            fields={fields}
            validationResult={validation}
            value={params}
            onChange={onChange}
         />
      </>
   );
};

storiesOf('forms/FormLayout', module).add('demo', () => <FormLayoutStory />);
