import * as React from 'react';
import { ComponentType } from 'react';

import { IValueProps } from '../../_models';
import { isEqual } from '../../helpers';
import { autobind, bindAndMemoize, IValidationResult } from '../../utils';
// noinspection TypeScriptPreferShortImport,ES6PreferShortImport
import { BooleanField } from '../fields/BooleanField';
// noinspection TypeScriptPreferShortImport,ES6PreferShortImport
import { BooleanSwitcherField } from '../fields/BooleanSwitcherField';
// noinspection TypeScriptPreferShortImport,ES6PreferShortImport
import { DateField } from '../fields/DateField';
// noinspection TypeScriptPreferShortImport,ES6PreferShortImport
import { DateTimeField } from '../fields/DateTimeField';
// noinspection TypeScriptPreferShortImport,ES6PreferShortImport
import { EnumField } from '../fields/EnumField';
// noinspection TypeScriptPreferShortImport,ES6PreferShortImport
import { HiddenField } from '../fields/HiddenField';
// noinspection TypeScriptPreferShortImport,ES6PreferShortImport
import { NumberField } from '../fields/NumberField';
// noinspection TypeScriptPreferShortImport,ES6PreferShortImport
import { NumberSetField } from '../fields/NumberSetField';
// noinspection TypeScriptPreferShortImport,ES6PreferShortImport
import { StringField } from '../fields/StringField';
// noinspection TypeScriptPreferShortImport,ES6PreferShortImport
import { StringSetField } from '../fields/StringSetField';
// noinspection TypeScriptPreferShortImport,ES6PreferShortImport
import { TextField } from '../fields/TextField';
import { FormFieldType, IFieldProps, IFormField } from '../models';

import styles from './FormLayout.module.css';

interface IProps extends IValueProps<object> {
   context?: any;
   fields: IFormField<any>[];
   readonly?: boolean;
   validationResult?: IValidationResult<any>;
}

type FieldTypesMap = { [key in FormFieldType]?: ComponentType<IFieldProps<any>> };

export class FormLayout extends React.PureComponent<IProps> {
   public static defaultProps = {
      context: null,
      readonly: false,
   };

   private fieldTypeComponents: FieldTypesMap = {
      [FormFieldType.Boolean]: BooleanField,
      [FormFieldType.BooleanSwitcher]: BooleanSwitcherField,
      [FormFieldType.Date]: DateField,
      [FormFieldType.DateTime]: DateTimeField,
      [FormFieldType.Enum]: EnumField,
      [FormFieldType.Hidden]: HiddenField,
      [FormFieldType.Number]: NumberField,
      [FormFieldType.NumberSet]: NumberSetField,
      [FormFieldType.String]: StringField,
      [FormFieldType.StringSet]: StringSetField,
      [FormFieldType.Text]: TextField,
   };

   public render() {
      const fields = this.props.fields.map(this.renderField);

      return <div className={styles.formLayout}>{fields}</div>;
   }

   private getValidationError(field: IFormField): string | null {
      const fieldName = field.validationField || field.name;
      const vr = this.props.validationResult;

      return vr && vr.errors.has(fieldName) ? vr.errors.get(fieldName)!.join(', ') : null;
   }

   @bindAndMemoize
   private onFieldChange(field: IFormField) {
      return (e: React.SyntheticEvent, value: any): void => {
         if (isEqual(this.props.value[field.name], value)) {
            return;
         }

         const params = {
            ...this.props.value,
            [field.name]: value,
         };
         this.props.onChange(e, params);
      };
   }

   @autobind
   private renderField(field: IFormField) {
      const value = this.props.value[field.name];
      const onChange = this.onFieldChange(field);

      if (field.isHidden && field.isHidden(this.props.value, this.props.context)) {
         return null;
      }

      const Field = field.type === 'custom' ? field.component : this.fieldTypeComponents[field.type];

      if (!Field) {
         return (
            <div key={field.name}>
               Unsupported field type <code>{field.type}</code>
            </div>
         );
      }

      const disabled = field.isDisabled ? field.isDisabled(this.props.value, this.props.context) : false;

      const help = field.getHelp ? field.getHelp(this.props.value, this.props.context) : null;

      const error = field.getError ? field.getError(this.props.value, this.props.context) : null;

      const validationError = this.getValidationError(field);

      return (
         <Field
            key={field.name}
            context={this.props.context}
            config={field}
            disabled={disabled}
            readonly={this.props.readonly!}
            help={help}
            error={error || validationError}
            formValue={this.props.value}
            value={value}
            onChange={onChange}
         />
      );
   }
}
