# Модуль форм

Содержит поля-обёртки над компонентами лего, немного оформленные для использования в формах с вертикальными полями.
Также средства для быстрого создания кастомных полей с таким же оформлением.

## Принципы, заложенные в модуль форм

1. Быть как можно ближе к [Formik](https://jaredpalmer.com/formik/docs/overview), минимум своего. Его нужно дополнить
   новыми возможностями, а не писать замену, полностью скрывая его в кишках.
2. Создавать кастомные поля нужно минимумом кода
3. Некоторые аспекты внешнего вида можно настроить через CSS-переменные

## Оформление формы

### Примеры полей

![Поле в обычном состоянии](images/inputField.png)

![Поле с ошибкой](images/inputFieldWithError.png)

### FieldLayout

Каждый контрол из представленных в библиотеке обернут в компонент <FieldLayout/>, который обеспечивает общую разметку,
оформление и логику отображения для всех типов полей.

FieldLayout необязателен, но рекомендован, для использования в кастомных типах полей.

#### Зоны `FieldLayout`

![Зоны поля](images/fieldZones.png)

У каждого поля можно выделить 4 зоны:

1. `label` - подпись к полю. Также может содержать красную звёздочку - индикатор обязательного поля.
2. `control` - сам контрол (поля для ввода информации)
3. `hint` - зона под label для отображения **краткого** пояснения к подписи. Если подпись слишком длинная, то она
   превращается в иконку с вопросиком, при нажатии на который выскочит попап.
4. `help` - зона для отображения пояснения к полю (например, списка возможных значений). Также в этой зоне отображается
   текст валидационной ошибки. В этом случае help пропадает (у него ниже приоритет, чем у ошибки).

#### Кастомизация `FieldLayout`

Тут всё просто. Базовые размеры задаются CSS-переменными. Названия вполне очевидны, пояснения не требуют.

```css
:root {
   --forms-label-width: 12.4rem;
   --forms-control-width: 22.6rem; /* input, textarea */
   --forms-small-control-width: 13.5rem; /* datepicker */
   --forms-invalid-control-bg: rgba(255, 0, 0, 0.05);
   --forms-invalid-placeholder-color: rgba(255, 0, 0, 0.56); /* 0.56 это как #999 только для красного */
}
```

## Примеры форм

Смотрите в разделе `Demo` сбоку. Рекомендую использовать декларативный подход, как самый простой.

**Временно код этих форм доступен только в репозитории. Я еще не допилил поддержку в storybook.**

## Поля из-коробки

| Название         | Назначение                                                                                            | Что под капотом               | controlProps                                       |
| :--------------- | :---------------------------------------------------------------------------------------------------- | :---------------------------- | :------------------------------------------------- |
| `InputField`     | Ввод коротких строк, чисел, email, паролей                                                            | `TextInput` от Лего           | type: 'text', 'email', 'tel', 'password', 'number' |
| `TextField`      | Ввод многострочного текста                                                                            | `AutoGrowTextArea` от ui-libs |                                                    |
| `CheckboxField`  | Просто галочка. Может отображать неопределённое значение (`null` или `undefined`).                    | `Checkbox` от Лего            |                                                    |
| `SwitcherField`  | Аналогичен `CheckboxField`, только вместо галочки переключатель. Не поддерживает `null` и `undefined` | `Tumbler` от Лего             |                                                    |
| `EnumField`      | Выбор из небольшого списка вариантов                                                                  | `RadioButton` от Лего         | options (см справку по полю)                       |
| `DateTimeField`  | Выбор даты или даты со временем                                                                       | `DateTimePicker` от ui-libs   | булевые флажки: showTimeString, withTime           |
| `StringSetField` | Портирование в планах                                                                                 | `AutoGrowTextArea` от ui-libs |                                                    |
| `NumberSetField` | Портирование в планах                                                                                 | `AutoGrowTextArea` от ui-libs |                                                    |
| `xxxField`       | Предлагайте варианты, добавим!                                                                        |                               |                                                    |

## Кастомные поля

В каждом проекте может возникнуть необходимость использовать в формат поля, специфичные для проекта.

Модуль спроектирован так, что нет никакой разницы между полями из-коробки и кастомными.

При определении формы мы в любом случае задаём компонент, который будет обрабатывать поле.

Технически, можно даже вообще не выносить поле в компонент, а просто использовать его разметку прямо в поле.

### Минимальное кастомное поле (без разметки вообще)

**Минимальное** поле должно удовлетворять следующему API (списку props):

```typescript
interface MinimalFieldProps<V> {
   /**
    * поле в модели формы
    */
   name: string;

   /**
    * актуальное значение в поле
    */
   value: V;

   /**
    * функция, вызываемая при обновления значения изнутри поля (уведомление формы об изменениях)
    */
   onChange(e: MinimalChangeEvent<V>): void;

   /**
    * функция, вызываемая при уходе фокуса с формы.
    *
    * Когда это неактуально, можно её вызывать при первом изменении значения.
    * Требуется для правильной работы валидации.
    */
   onBlur(e: MinimalBlurEvent): void;
}

interface MinimalChangeEvent<V> {
   target: {
      /**
       * совпадает с `name` переданным полю
       */
      name: string;
      /**
       * Новое значение
       */
      value: V;
   };
}

interface MinimalBlurEvent {
   target: {
      /**
       * совпадает с `name` переданным полю
       */
      name: string;
   };
}
```

Таким образом минимальный компонент кастомного поля может выглядеть так:

```typescript jsx
function CustomInputField({ name, value, onChange, onBlur }: MinimalFieldProps<string>) {
   return <input type='text' name={name} value={value} onChange={onChange} onBlur={onBlur} />;
}

// Или еще проще:

function CustomInputField(props: MinimalFieldProps<string>) {
   return <input type='text' {...props} />;
}
```

Но следует учитывать, что это поле будет без разметки.

### Рекомендуемый подход к созданию кастомного поле

Если же хочется более единообразного вида (как прочие поля в форме) минимумом усилий:

```typescript jsx
function CustomInputField(props: ExtendedFieldConfig) {
   const {
      field,
      readonly, // Вычисляемое значения
      disabled, // Вычисляемое значения
      onChange,
      onBlur,
      showError,
   } = useExtendedField(props);

   return (
      <FieldLayout {...props}>
         <input
            type='text'
            id={field.name} // id удобен для работы <label for="id"/>, но необязателен
            name={field.name}
            value={field.value}
            onChange={onChange}
            onBlur={onBlur}
            placeholder={props.placeholder}
            disabled={disabled}
            readOnly={readonly}
         />
      </FieldLayout>
   );
}
```

Теперь это поле получит поддержку:

-  подписей,
-  справочных пояснений как к полю, так и к label
-  вывод валидационной ошибки,
-  индикатора обязательности и т.д.
-  placeholder
-  поддержку статусов disabled и readonly

## Справочник API для формы

TODO

## Справочник API для полей

TODO

## Примеры простых форм

TODO

## Примеры сложных форм

TODO ...в том числе вложенных форм и массивов субформ.
