import { useField, useFormikContext } from 'formik';
import React, { useContext, useMemo } from 'react';

import { classNames } from '../../../formatters';
import { isEmpty } from '../../../helpers';
import { Rows } from '../../../small_components';
import { getCalculatedValue } from '../../helpers';
import { ExtendedFieldConfig, FormContext } from '../../models';
import { OverridingFieldPropsContext } from '../../wrappers';
import { FieldError } from '../FieldError/FieldError';

import classes from './FieldLayout.module.css';
import { LabelHint } from './LabelHint/LabelHint';

export const FieldLayout: React.FC<ExtendedFieldConfig> = React.memo(
   ({ bigLabel = false, hideErrors = false, readonlyDots = false, ...rawProps }) => {
      const overridingProps = useContext(OverridingFieldPropsContext);
      const props = useMemo(() => ({ ...rawProps, ...overridingProps }), [overridingProps, rawProps]);

      const formContext = useContext(FormContext);
      const form = useFormikContext();
      const [field, meta] = useField(props as any); // TODO remove any
      const qa = props.qa ?? props.name.toString();

      // region render
      const hidden = getCalculatedValue(props.hidden, form, field, formContext);
      if (hidden === true) {
         return null;
      }

      const help = getCalculatedValue(props.help, form, field, formContext);
      let message: JSX.Element | null = null;
      const showError = !hideErrors && meta.error && meta.touched;

      if (showError && !isEmpty(meta.error)) {
         message = (
            <div className={classes.error} data-qa={`${qa}:FieldLayout/error`}>
               <FieldError>{meta.error as any}</FieldError>
            </div>
         );
      } else if (help) {
         message = (
            <div className={classes.help} data-qa={`${qa}:FieldLayout/help`}>
               {typeof help === 'string' ? <Rows text={help} /> : help}
            </div>
         );
      }

      const { label } = props;

      const readonly = getCalculatedValue(props.readonly, form, field, formContext);

      const required = !readonly && getCalculatedValue(props.required, form, field, formContext);

      const showDots = readonly && readonlyDots;

      return (
         <div
            className={classNames(
               classes.field,
               { [classes.bigLabelField]: bigLabel, [classes.readonlyDotsField]: showDots },
               props.className,
            )}
            data-qa={`${qa}:FieldLayout`}
         >
            {label === null ? null : label === undefined ? (
               <label
                  className={classNames(classes.label, props.labelClassName)}
                  data-qa={`${qa}:FieldLayout/label`}
                  htmlFor={field.name}
               />
            ) : (
               <label
                  className={classNames(
                     classes.label,
                     {
                        [classes.bigLabel]: bigLabel,
                        [classes.dotsLabel]: showDots,
                        [classes.labelError]: showError,
                     },
                     props.labelClassName,
                  )}
                  data-qa={`${qa}:FieldLayout/label`}
                  htmlFor={field.name}
               >
                  {typeof label === 'string' ? <span className={classes.labelText}>{label}</span> : label}
                  {required && (
                     <>
                        &#65279;<span className={classes.required}>{required ? '*' : <>&#65279;</>}</span>
                     </>
                  )}
                  &nbsp;
                  <LabelHint
                     forceHint={showDots}
                     hintClassName={classes.hint}
                     qa={`${qa}:FieldLayout/hint`}
                     smallClassName={classes.smallHint}
                     text={props.hint}
                  />
               </label>
            )}

            <div className={classes.input} data-qa={`${qa}:FieldLayout/input`}>
               {props.children}
               {message}
            </div>
         </div>
      );
      // endregion
   },
);

FieldLayout.displayName = 'FieldLayout';
